<?php

namespace App\Exports;

use App\Models\Client;
use App\Models\Conveyance;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;

class ClientEmployeeConveyanceExport implements FromCollection, WithHeadings, ShouldAutoSize
{
    protected Client $client;
    protected Carbon $monthStart;
    protected Carbon $monthEnd;
    protected string $monthLabel;

    /**
     * @param Client $client
     * @param Carbon $monthStart
     * @param Carbon $monthEnd
     * @param string $monthLabel
     */
    public function __construct(Client $client, Carbon $monthStart, Carbon $monthEnd, string $monthLabel)
    {
        $this->client     = $client;
        $this->monthStart = $monthStart;
        $this->monthEnd   = $monthEnd;
        $this->monthLabel = $monthLabel;
    }

    /**
     * Excel headings
     */
    public function headings(): array
    {
        return [
            'Client',
            'Month',
            'Employee Name',
            'Employee Email',
            'Date',
            'From',
            'To',
            'Mode',
            'Onward (₹)',
            'Return (₹)',
            'Total (₹)',
            'Remarks',
            'Return Remarks',
            'Status',
        ];
    }

    /**
     * Data for Excel
     */
    public function collection()
    {
        $rows = Conveyance::with(['user', 'client'])
            ->where('client_id', $this->client->id)
            ->whereBetween('date', [
                $this->monthStart->toDateString(),
                $this->monthEnd->toDateString(),
            ])
            ->where('status', 'submitted')
            ->orderBy('user_id')
            ->orderBy('date')
            ->get();

        $data = collect();
        $grandTotal = 0;

        foreach ($rows as $c) {
            $user  = $c->user;
            $name  = $user->name ?? 'Unknown';
            $email = $user->email ?? '';

            $onward = (float) $c->amount;

            if ($c->is_return) {
                $return = $c->return_amount !== null
                    ? (float) $c->return_amount
                    : $onward;
            } else {
                $return = 0.0;
            }

            $rowTotal   = $onward + $return;
            $grandTotal += $rowTotal;

            $data->push([
                $this->client->name ?? 'Unknown Client',
                $this->monthLabel,
                $name,
                $email,
                $c->date->format('d-m-Y'),
                $c->from_location,
                $c->to_location,
                $c->mode ?? '-',
                number_format($onward, 2, '.', ''),
                $c->is_return ? number_format($return, 2, '.', '') : '',
                number_format($rowTotal, 2, '.', ''),
                $c->remarks ?: '',
                $c->return_remarks ?: '',
                $c->status,
            ]);
        }

        // Grand total row at bottom
        if ($data->isNotEmpty()) {
            $data->push([
                '',
                '',
                '',
                '',
                '',
                '',
                '',
                'GRAND TOTAL',
                '',
                '',
                number_format($grandTotal, 2, '.', ''),
                '',
                '',
                '',
            ]);
        }

        return $data;
    }
}
