<?php

namespace App\Exports;

use App\Models\Client;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\WithMultipleSheets;

class ClientEmployeeDetailedExport implements WithMultipleSheets
{
    protected Client $client;
    protected ?Carbon $monthStart;
    protected ?Carbon $monthEnd;
    protected string $monthLabel;
    protected ?int $sessionId;

    public function __construct(Client $client, ?Carbon $monthStart, ?Carbon $monthEnd, string $monthLabel, ?int $sessionId = null)
    {
        $this->client     = $client;
        $this->monthStart = $monthStart;
        $this->monthEnd   = $monthEnd;
        $this->monthLabel = $monthLabel;
        $this->sessionId  = $sessionId;
    }

    /**
     * @return array
     */
    public function sheets(): array
    {
        $sheets = [];

        // 1. First sheet: Summary
        // We can reuse the existing summary logic or create a dedicated summary sheet class. 
        // For "advanced" feel, let's create a Master Summary sheet first.
        $sheets[] = new ClientEmployeeConveyanceExport(
            $this->client, 
            $this->monthStart ?? Carbon::create(2000,1,1), // fallback if null, though controller ensures logic
            $this->monthEnd   ?? Carbon::now()->addYears(10),
            $this->monthLabel
        );
        // Rename the summary sheet to 'Master Summary' (requires modifying ClientEmployeeConveyanceExport or wrapping it)
        // Actually, ClientEmployeeConveyanceExport doesn't implement WithTitle. Let's make a specific Sheet class.

        // Get all unique users who have data for this client in this range
        $query = \App\Models\Conveyance::where('client_id', $this->client->id)
            ->where('status', 'submitted');

        if ($this->monthStart && $this->monthEnd) {
            $query->whereBetween('date', [$this->monthStart->toDateString(), $this->monthEnd->toDateString()]);
        } elseif ($this->monthStart) {
            $query->where('date', '>=', $this->monthStart->toDateString());
        } elseif ($this->monthEnd) {
            $query->where('date', '<=', $this->monthEnd->toDateString());
        }
        
        if ($this->sessionId) {
            $query->where('session_id', $this->sessionId);
        }

        // distinct users
        $userIds = $query->distinct()->pluck('user_id');
        $users = \App\Models\User::whereIn('id', $userIds)->orderBy('name')->get();

        foreach ($users as $user) {
            $sheets[] = new ClientEmployeeSheetExport(
                $user,
                $this->client,
                $this->monthStart,
                $this->monthEnd,
                $this->sessionId
            );
        }

        return $sheets;
    }
}
