<?php

namespace App\Exports;

use App\Models\User;
use App\Models\Client;
use App\Models\Conveyance;
use Carbon\Carbon;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class ClientEmployeeSheetExport implements FromCollection, WithHeadings, WithTitle, ShouldAutoSize, WithStyles
{
    protected User $user;
    protected Client $client;
    protected ?Carbon $monthStart;
    protected ?Carbon $monthEnd;
    protected ?int $sessionId;

    public function __construct(User $user, Client $client, ?Carbon $monthStart, ?Carbon $monthEnd, ?int $sessionId = null)
    {
        $this->user = $user;
        $this->client = $client;
        $this->monthStart = $monthStart;
        $this->monthEnd = $monthEnd;
        $this->sessionId = $sessionId;
    }

    public function title(): string
    {
        // Sheet name (Excel has 31 char limit)
        return substr($this->user->name, 0, 30);
    }

    public function headings(): array
    {
        return [
            'Date',
            'From',
            'To',
            'Mode',
            'Onward (₹)',
            'Return (₹)',
            'Total (₹)',
            'Remarks',
            'Return Remarks',
        ];
    }

    public function collection()
    {
        $query = Conveyance::where('client_id', $this->client->id)
            ->where('user_id', $this->user->id)
            ->where('status', 'submitted');

        if ($this->monthStart && $this->monthEnd) {
            $query->whereBetween('date', [$this->monthStart->toDateString(), $this->monthEnd->toDateString()]);
        } elseif ($this->monthStart) {
            $query->where('date', '>=', $this->monthStart->toDateString());
        } elseif ($this->monthEnd) {
            $query->where('date', '<=', $this->monthEnd->toDateString());
        }

        if ($this->sessionId) {
            $query->where('session_id', $this->sessionId);
        }

        $rows = $query->orderBy('date')->get();

        $data = collect();
        $grandTotal = 0;

        foreach ($rows as $c) {
            $onward = (float) $c->amount;
            if ($c->is_return) {
                $return = $c->return_amount !== null ? (float) $c->return_amount : $onward;
            } else {
                $return = 0.0;
            }

            $rowTotal = $onward + $return;
            $grandTotal += $rowTotal;

            $data->push([
                $c->date->format('d-m-Y'),
                $c->from_location,
                $c->to_location,
                $c->mode ?? '-',
                number_format($onward, 2, '.', ''),
                $c->is_return ? number_format($return, 2, '.', '') : '',
                number_format($rowTotal, 2, '.', ''),
                $c->remarks ?: '',
                $c->return_remarks ?: '',
            ]);
        }

        // Add Total Row
        if ($data->isNotEmpty()) {
            $data->push([
                '', '', '', 'TOTAL',
                '', '',
                number_format($grandTotal, 2, '.', ''),
                '', ''
            ]);
        }

        return $data;
    }

    public function styles(Worksheet $sheet)
    {
        // Bold headings
        $sheet->getStyle('A1:I1')->getFont()->setBold(true);

        // Bold last row (Total)
        $lastRow = $sheet->getHighestRow();
        $sheet->getStyle('A' . $lastRow . ':I' . $lastRow)->getFont()->setBold(true);
        
        // Color the total cell green
        $sheet->getStyle('G' . $lastRow)->getFont()->getColor()->setARGB('FF008000');
    }
}
