<?php

namespace App\Exports;

use App\Models\User;
use Illuminate\Contracts\View\View;
use Maatwebsite\Excel\Concerns\WithMultipleSheets;
use Maatwebsite\Excel\Concerns\FromView;
use Maatwebsite\Excel\Concerns\WithTitle;

class EmployeeConveyanceMonthFullExport implements WithMultipleSheets
{
    public User $employee;
    public array $monthsData;

    public function __construct(User $employee, array $monthsData)
    {
        $this->employee   = $employee;
        $this->monthsData = $monthsData;
    }

    /**
     * One workbook:
     *  - Sheet 1: Month summary grid (like UI, month vs days)
     *  - Sheet N: One sheet per month with full day-wise entries
     */
    public function sheets(): array
    {
        $sheets = [];

        // 1) SUMMARY SHEET (month rows, day columns)
        // Compute maxDays like in EmployeeConveyanceMonthSummaryExport
        $maxDays = 0;
        foreach ($this->monthsData as $month) {
            $maxDays = max($maxDays, $month['days_in_month']);
        }
        if ($maxDays === 0) {
            $maxDays = 31;
        }

        $sheets[] = new class($this->employee, $this->monthsData, $maxDays) implements FromView, WithTitle {
            public User $employee;
            public array $monthsData;
            public int $maxDays;

            public function __construct(User $employee, array $monthsData, int $maxDays)
            {
                $this->employee   = $employee;
                $this->monthsData = $monthsData;
                $this->maxDays    = $maxDays;
            }

            public function view(): View
            {
                return view('admin.conveyance.exports.employee_month_summary', [
                    'employee'   => $this->employee,
                    'monthsData' => $this->monthsData,
                    'maxDays'    => $this->maxDays,
                ]);
            }

            public function title(): string
            {
                return 'Summary';
            }
        };

        // 2) DETAILED SHEETS – one per month
        foreach ($this->monthsData as $month) {
            $sheets[] = new class($this->employee, $month) implements FromView, WithTitle {
                public User $employee;
                public array $month;

                public function __construct(User $employee, array $month)
                {
                    $this->employee = $employee;
                    $this->month    = $month;
                }

                public function view(): View
                {
                    return view('admin.conveyance.exports.employee_month_full', [
                        'employee' => $this->employee,
                        'month'    => $this->month,
                    ]);
                }

                public function title(): string
                {
                    // Sheet name like "Nov 2025"
                    return $this->month['start']->format('M Y');
                }
            };
        }

        return $sheets;
    }
}
