<?php

namespace App\Exports;

use App\Models\Conveyance;
use App\Models\Session;
use App\Models\User;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\FromView;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Illuminate\Contracts\View\View;

class EmployeeMonthlyConveyanceExport implements FromView, ShouldAutoSize
{
    protected int $sessionId;

    public function __construct(int $sessionId)
    {
        $this->sessionId = $sessionId;
    }

    public function view(): View
    {
        $activeSession = Session::findOrFail($this->sessionId);

        $sessionStart = Carbon::parse($activeSession->start_date)->startOfMonth();
        $sessionEnd = Carbon::parse($activeSession->end_date)->endOfMonth();

        $months = [];
        $cursor = $sessionStart->copy();
        while ($cursor->lte($sessionEnd)) {
            $key = $cursor->format('Y-m');
            $months[] = [
                'key' => $key,
                'label' => $cursor->format('M Y'),
            ];
            $cursor->addMonth();
        }

        // Raw data fetch - Get individual rows and aggregate in PHP to avoid SQL sum issues
        $raw = Conveyance::with('user')
            ->whereBetween('date', [$sessionStart->toDateString(), $sessionEnd->toDateString()])
            ->where('session_id', $activeSession->id)
            ->where('status', 'submitted')
            ->get();

        // 1. Fetch ALL employees (users with role 'employee')
        // We want to show everyone, even if they have 0 totals.
        $users = User::where('role', 'employee')
            ->orderBy('name')
            ->get();

        // Prepare rows index
        $rows = [];
        foreach ($users as $user) {
            $rows[$user->id] = [
                'user' => $user,
                'monthly' => [],
                'grand_total' => 0,
            ];
        }

        $overallTotal = 0;

        // Fill monthly totals via PHP iteration
        foreach ($raw as $item) {
            $uid = $item->user_id;

            // Ensure user exists in our list (in case role changed or old data)
            if (!isset($rows[$uid])) {
                continue;
            }

            // Calculate amount for this specific row in PHP
            $rowAmount = (float) $item->amount;
            if ($item->is_return) {
                // Check if return_amount is present, otherwise use amount again
                $returnVal = ($item->return_amount !== null && $item->return_amount !== '')
                    ? (float) $item->return_amount
                    : (float) $item->amount;
                $rowAmount += $returnVal;
            }

            $ym = $item->date->format('Y-m');

            if (!isset($rows[$uid]['monthly'][$ym])) {
                $rows[$uid]['monthly'][$ym] = 0;
            }

            $rows[$uid]['monthly'][$ym] += $rowAmount;
            $rows[$uid]['grand_total'] += $rowAmount;
            $overallTotal += $rowAmount;
        }

        // We can reuse the same view but with a flag to hide buttons / extra UI
        return view('admin.reports.export_employee_monthly', [
            'activeSession' => $activeSession,
            'months' => $months,
            'rows' => $rows,
            'overallTotal' => $overallTotal,
        ]);

    }
}
