<?php

namespace App\Exports;

use App\Models\Expense;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Illuminate\Support\Carbon;

class ExpensesExport implements FromQuery, WithHeadings, WithMapping, ShouldAutoSize
{
    protected ?string $fromDate;
    protected ?string $toDate;
    protected ?int $employeeId;

    public function __construct(?string $fromDate = null, ?string $toDate = null, ?int $employeeId = null)
    {
        $this->fromDate   = $fromDate;
        $this->toDate     = $toDate;
        $this->employeeId = $employeeId;
    }

    public function query()
    {
        $q = Expense::query()->with(['user', 'employee']);

        if ($this->fromDate) {
            $q->whereDate('expense_date', '>=', $this->fromDate);
        }

        if ($this->toDate) {
            $q->whereDate('expense_date', '<=', $this->toDate);
        }

        if ($this->employeeId) {
            $q->where('user_id', $this->employeeId);
        }

        return $q->orderBy('expense_date', 'asc');
    }

    public function headings(): array
    {
        return [
            'ID',
            'Expense Date',
            'Employee Name',
            'Employee Email',
            'Group',
            'Party Name',
            'Purpose',
            'Amount',
            'Session',
            'Created At',
        ];
    }

    public function map($expense): array
    {
        return [
            $expense->id,
            optional($expense->expense_date)->format('Y-m-d'),
            optional($expense->user)->name,
            optional($expense->user)->email,
            $expense->group,
            $expense->party_name,
            $expense->purpose,
            number_format((float)$expense->amount, 2, '.', ''),
            optional($expense->financialSession)->name ?? '', // if you have relation
            optional($expense->created_at)->format('Y-m-d H:i:s'),
        ];
    }
}
