<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Session;
use App\Models\Employee;
use App\Models\Conveyance;
use Carbon\Carbon;

class AdminDashboardController extends Controller
{
    public function index()
    {
        $activeSession = Session::active();
        $totalSessions = Session::count();
        $totalEmployees = Employee::count();
        $employeesInActiveSession = $activeSession ? $activeSession->employees()->count() : 0;

        $today = Carbon::today();

        // Claim month logic (same as employee side)
        if ($today->day <= 10) {
            $targetMonth = $today->copy()->subMonthNoOverflow();
        } else {
            $targetMonth = $today->copy();
        }

        $monthStart = $targetMonth->copy()->startOfMonth();
        $monthEnd = $targetMonth->copy()->endOfMonth();
        $currentMonthLabel = $targetMonth->format('F Y');

        // Pull all submitted conveyance rows for this month
        $conveyances = Conveyance::whereBetween('date', [$monthStart, $monthEnd])
            ->where('status', 'submitted')
            ->with('client')
            ->get();

        // Row-total logic (same as Employee controller)
        $rowTotal = function ($c) {
            $onward = (float) $c->amount;

            if ($c->is_return) {
                $return = $c->return_amount !== null
                    ? (float) $c->return_amount
                    : $onward;

                return $onward + $return;
            }

            return $onward;
        };

        /** -------------------------------
         *  ADMIN DASHBOARD METRICS (FIXED)
         * --------------------------------*/

        // Total conveyance amount this month (correct calculation)
        $totalConveyanceThisMonth = $conveyances->sum($rowTotal);

        // Unique clients count
        $clientsThisMonth = $conveyances
            ->whereNotNull('client_id')
            ->groupBy('client_id')
            ->count();

        // Client summaries (cards)
        $clientSummaries = $conveyances
            ->whereNotNull('client_id')
            ->groupBy('client_id')
            ->map(function ($items) use ($rowTotal) {

                $client = $items->first()->client;

                return [
                    'client_id' => $client?->id,
                    'name' => $client?->name ?? 'No Client',
                    'city' => $client?->city ?? null,
                    'total_amount' => $items->sum($rowTotal),   // FIXED total
                    'entry_count' => $items->count(),
                    'employee_count' => $items->groupBy('user_id')->count(),
                ];
            })
            ->values(); // reset keys

        /** -------------------------------
         *  INSIGHTS (MONTHLY & ALL TIME)
         * --------------------------------*/

        // Helper for payout calculation raw SQL
        $payoutSql = "SUM(CAST(amount AS DECIMAL(10,2)) + CASE WHEN is_return = 1 THEN COALESCE(CAST(return_amount AS DECIMAL(10,2)), CAST(amount AS DECIMAL(10,2))) ELSE 0 END)";

        // 1. Top 5 Employees (This Month)
        $topEmployeesMonth = Conveyance::where('status', 'submitted')
            ->whereBetween('date', [$monthStart, $monthEnd])
            ->selectRaw("user_id, $payoutSql as total_payout")
            ->groupBy('user_id')
            ->orderByDesc('total_payout')
            ->limit(5)
            ->with('user')
            ->get();

        // 2. Top 5 Clients (This Month)
        $topClientsMonth = Conveyance::where('status', 'submitted')
            ->whereBetween('date', [$monthStart, $monthEnd])
            ->whereNotNull('client_id')
            ->selectRaw("client_id, $payoutSql as total_payout")
            ->groupBy('client_id')
            ->orderByDesc('total_payout')
            ->limit(5)
            ->with('client')
            ->get();

        // 3. Top 5 Employees (All Time)
        $topEmployees = Conveyance::where('status', 'submitted')
            ->selectRaw("user_id, $payoutSql as total_payout")
            ->groupBy('user_id')
            ->orderByDesc('total_payout')
            ->limit(5)
            ->with('user')
            ->get();

        // 4. Top 5 Clients (All Time)
        $topClients = Conveyance::where('status', 'submitted')
            ->whereNotNull('client_id')
            ->selectRaw("client_id, $payoutSql as total_payout")
            ->groupBy('client_id')
            ->orderByDesc('total_payout')
            ->limit(5)
            ->with('client')
            ->get();

        return view('admin.dashboard', compact(
            'activeSession',
            'totalSessions',
            'totalEmployees',
            'employeesInActiveSession',
            'currentMonthLabel',
            'monthStart',
            'monthEnd',
            'totalConveyanceThisMonth',
            'clientsThisMonth',
            'clientSummaries',
            'topEmployees',
            'topClients',
            'topEmployeesMonth',
            'topClientsMonth'
        ));
    }
}
