<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Conveyance;
use App\Models\Session;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\EmployeeMonthlyConveyanceExport;

class EmployeeMonthlyConveyanceController extends Controller
{
    /**
     * Show Employee × Month-wise consolidated conveyance (submitted only).
     */
    public function index(Request $request)
    {
        $activeSession = Session::active();
        if (!$activeSession) {
            abort(403, 'No active financial session found.');
        }

        // Use session start/end to define the months we show
        $sessionStart = Carbon::parse($activeSession->start_date)->startOfMonth();
        $sessionEnd = Carbon::parse($activeSession->end_date)->endOfMonth();

        // Build month list (Apr 2025, May 2025, etc…)
        $months = [];
        $cursor = $sessionStart->copy();
        while ($cursor->lte($sessionEnd)) {
            $key = $cursor->format('Y-m');   // for matching in query
            $months[] = [
                'key' => $key,
                'label' => $cursor->format('M Y'),
            ];
            $cursor->addMonth();
        }

        // Raw data fetch - Get individual rows and aggregate in PHP to avoid SQL sum issues
        $raw = Conveyance::with('user')
            ->whereBetween('date', [$sessionStart->toDateString(), $sessionEnd->toDateString()])
            ->where('session_id', $activeSession->id)
            ->where('status', 'submitted')
            ->get();

        // 1. Fetch ALL employees (users with role 'employee') (Reordered logic slightly)
        $users = User::where('role', 'employee')
            ->orderBy('name')
            ->get();
        // Prepare rows index
        $rows = [];
        foreach ($users as $user) {
            $rows[$user->id] = [
                'user' => $user,
                'monthly' => [],
                'grand_total' => 0,
            ];
        }

        $overallTotal = 0;

        // Fill monthly totals via PHP iteration
        foreach ($raw as $item) {
            $uid = $item->user_id;

            // Ensure user exists in our list (in case role changed or old data)
            if (!isset($rows[$uid])) {
                continue;
            }

            // Calculate amount for this specific row in PHP
            $rowAmount = (float) $item->amount;
            if ($item->is_return) {
                // Check if return_amount is present, otherwise use amount again
                $returnVal = ($item->return_amount !== null && $item->return_amount !== '')
                    ? (float) $item->return_amount
                    : (float) $item->amount;
                $rowAmount += $returnVal;
            }

            $ym = $item->date->format('Y-m');

            if (!isset($rows[$uid]['monthly'][$ym])) {
                $rows[$uid]['monthly'][$ym] = 0;
            }

            $rows[$uid]['monthly'][$ym] += $rowAmount;
            $rows[$uid]['grand_total'] += $rowAmount;
            $overallTotal += $rowAmount;
        }

        return view('admin.reports.employee_monthly_consolidated', [
            'activeSession' => $activeSession,
            'months' => $months,
            'rows' => $rows,
            'overallTotal' => $overallTotal,
        ]);
    }

    /**
     * Download the same Employee × Month-wise table as Excel.
     */
    public function export(Request $request)
    {
        $activeSession = Session::active();
        if (!$activeSession) {
            abort(403, 'No active financial session found.');
        }

        $fileName = 'employee-monthly-conveyance-' . now()->format('Ymd_His') . '.xlsx';

        return Excel::download(
            new EmployeeMonthlyConveyanceExport($activeSession->id),
            $fileName
        );
    }
}
