<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Expense;
use App\Models\User;
use App\Models\Session as FinancialSession;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\ExpensesExport;
use Illuminate\Support\Facades\Storage;

class ExpenseReportController extends Controller
{
    public function __construct()
    {
        // put your admin middleware here if needed
        // $this->middleware(['auth', 'is_admin']);
    }

    /**
     * Show expense report with filters + table.
     */
    public function index(Request $request)
    {
        // Filters
        $month = $request->input('month');       // format: 2025-11
        $fromDate = $request->input('from_date');
        $toDate = $request->input('to_date');
        $employeeId = $request->input('employee_id');

        // If month is selected and from/to empty, derive range from month
        if ($month && (!$fromDate && !$toDate)) {
            try {
                $start = Carbon::createFromFormat('Y-m', $month)->startOfMonth();
                $end = Carbon::createFromFormat('Y-m', $month)->endOfMonth();
                $fromDate = $start->toDateString();
                $toDate = $end->toDateString();
            } catch (\Throwable $e) {
            }
        }

        // Active session (for header badge)
        $activeSession = FinancialSession::where('is_active', true)->first();
        // Employees dropdown (only employees who have expenses) for filter
        $employees = User::whereHas('expenses')->orderBy('name')->get();

        // MODE SWITCH: Detail vs Summary
        if ($employeeId) {
            // --- DETAIL MODE ---
            $mode = 'detail';
            $selectedEmployee = User::find($employeeId);

            $query = Expense::query()
                ->with(['user', 'employee', 'session'])
                ->where('user_id', $employeeId)
                ->orderBy('expense_date', 'desc');

            if ($fromDate)
                $query->whereDate('expense_date', '>=', $fromDate);
            if ($toDate)
                $query->whereDate('expense_date', '<=', $toDate);

            $expenses = $query->paginate(50)->withQueryString();

            // Stats for this employee in this range
            $totalAmount = $expenses->sum('amount'); // This only sums current page, we might want total query sum
            // Better:
            $totalAmount = $query->sum('amount'); // This runs a separate count query effectively? No, sum query.
            // Note: $query was modified by paginate? paginate returns result, doesn't modify builder query in-place usually in a destructive way for sum if called before.
            // Actually paginate executes. We need a clone or separate calc.

            // Re-instantiate for totals to be accurate across pages
            $sumQuery = Expense::where('user_id', $employeeId);
            if ($fromDate)
                $sumQuery->whereDate('expense_date', '>=', $fromDate);
            if ($toDate)
                $sumQuery->whereDate('expense_date', '<=', $toDate);
            $totalAmount = $sumQuery->sum('amount');
            $totalRecords = $sumQuery->count();

            return view('admin.expenses.report', compact(
                'mode',
                'expenses',
                'selectedEmployee',
                'employees',
                'activeSession',
                'month',
                'fromDate',
                'toDate',
                'employeeId',
                'totalAmount',
                'totalRecords'
            ));

        } else {
            // --- SUMMARY MODE ---
            $mode = 'summary';

            // Fetch employees with aggregated expenses in range
            $summaryData = User::where('role', 'employee')
                ->whereHas('expenses', function ($q) use ($fromDate, $toDate) {
                    if ($fromDate)
                        $q->whereDate('expense_date', '>=', $fromDate);
                    if ($toDate)
                        $q->whereDate('expense_date', '<=', $toDate);
                })
                ->withSum([
                    'expenses' => function ($q) use ($fromDate, $toDate) {
                        if ($fromDate)
                            $q->whereDate('expense_date', '>=', $fromDate);
                        if ($toDate)
                            $q->whereDate('expense_date', '<=', $toDate);
                    }
                ], 'amount')
                ->withCount([
                    'expenses' => function ($q) use ($fromDate, $toDate) {
                        if ($fromDate)
                            $q->whereDate('expense_date', '>=', $fromDate);
                        if ($toDate)
                            $q->whereDate('expense_date', '<=', $toDate);
                    }
                ])
                ->orderByDesc('expenses_sum_amount')
                ->get();

            $totalAmount = $summaryData->sum('expenses_sum_amount');
            $totalRecords = $summaryData->sum('expenses_count');

            return view('admin.expenses.report', compact(
                'mode',
                'summaryData',
                'employees',
                'activeSession',
                'month',
                'fromDate',
                'toDate',
                'employeeId',
                'totalAmount',
                'totalRecords'
            ));
        }
    }

    /**
     * Export filtered expenses to Excel.
     */
    public function export(Request $request)
    {
        $month = $request->input('month');
        $fromDate = $request->input('from_date');
        $toDate = $request->input('to_date');
        $employeeId = $request->input('employee_id');

        if ($month && (!$fromDate && !$toDate)) {
            try {
                $start = Carbon::createFromFormat('Y-m', $month)->startOfMonth();
                $end = Carbon::createFromFormat('Y-m', $month)->endOfMonth();
                $fromDate = $start->toDateString();
                $toDate = $end->toDateString();
            } catch (\Throwable $e) {
                // ignore
            }
        }

        $filenameParts = ['expenses'];
        if ($fromDate && $toDate) {
            $filenameParts[] = $fromDate . '_to_' . $toDate;
        } elseif ($month) {
            $filenameParts[] = $month;
        }
        if ($employeeId) {
            $filenameParts[] = 'employee_' . $employeeId;
        }
        $filename = implode('_', $filenameParts) . '.xlsx';

        return Excel::download(
            new ExpensesExport($fromDate, $toDate, $employeeId),
            $filename
        );
    }

    public function downloadProof(Request $request, Expense $expense)
    {
        $file = $request->query('file');

        if (!$file) {
            abort(404, "File not specified.");
        }

        $availableFiles = [];

        if (is_array($expense->proof_paths)) {
            foreach ($expense->proof_paths as $pf) {
                $availableFiles[] = $pf['path'];
            }
        } elseif ($expense->proof_path) {
            $availableFiles[] = $expense->proof_path;
        }

        if (!in_array($file, $availableFiles)) {
            abort(404, "This attachment does not belong to this expense.");
        }

        $file = ltrim($file, "/");

        if (!Storage::disk('public')->exists($file)) {
            abort(404, "File not found in storage.");
        }

        return Storage::disk('public')->download($file);
    }

}
