<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Mail\SendOtpMail;
use App\Models\Employee;
use App\Models\OtpCode;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;

class OtpAuthController extends Controller
{
    public function showLogin()
    {
        return view('auth.login');
    }

    public function sendOtp(Request $request)
    {
        $request->validate(['identifier' => 'required']);

        $identifier = trim($request->identifier);
        $channel    = filter_var($identifier, FILTER_VALIDATE_EMAIL) ? 'email' : 'phone';

        // CLOSED SYSTEM:
        // employee master OR admin user allowed
        $employee = Employee::where($channel, $identifier)
            ->where('is_active', true)
            ->first();

        $adminUser = User::where($channel, $identifier)
            ->where('role', 'admin')
            ->first();

        if (!$employee && !$adminUser) {
            return back()->withErrors([
                'identifier' => 'Not registered with PAC Technology. Contact admin.',
            ]);
        }

        // phone OTP disabled currently
        if ($channel === 'phone' && !config('otp.phone_enabled')) {
            return back()->withErrors([
                'identifier' => 'Phone OTP disabled. Use Email OTP.',
            ]);
        }

        $otp = rand(100000, 999999);

        OtpCode::create([
            'identifier' => $identifier,
            'channel'    => $channel,
            'code'       => $otp,
            'expires_at' => now()->addMinutes(5),
        ]);

        if ($channel === 'email') {
            Mail::to($identifier)->send(new SendOtpMail($otp));
        }

        session([
            'otp_identifier' => $identifier,
            'otp_channel'    => $channel,
        ]);

        return redirect()->route('otp.verify.form')
            ->with('success', 'OTP sent.');
    }

    public function showVerifyForm()
    {
        if (!session('otp_identifier')) {
            return redirect()->route('login');
        }

        return view('auth.verify');
    }

    public function verifyOtp(Request $request)
    {
        $request->validate(['otp' => 'required|digits:6']);

        $identifier = session('otp_identifier');
        $channel    = session('otp_channel');

        $record = OtpCode::where('identifier', $identifier)
            ->where('channel', $channel)
            ->whereNull('used_at')
            ->latest()
            ->first();

        if (!$record || now()->gt($record->expires_at)) {
            return back()->withErrors(['otp' => 'OTP expired. Resend OTP.']);
        }

        if ($record->code !== $request->otp) {
            $record->increment('attempts');
            return back()->withErrors(['otp' => 'Invalid OTP.']);
        }

        $record->update(['used_at' => now()]);

        // 1) ADMIN LOGIN (as-is)
        $adminUser = User::where($channel, $identifier)
            ->where('role', 'admin')
            ->first();

        if ($adminUser) {
            Auth::login($adminUser);
            session()->forget(['otp_identifier', 'otp_channel']);

            return redirect()->route('admin.dashboard');
        }

        // 2) EMPLOYEE LOGIN (fixed, no duplicate email)
        $employee = Employee::where($channel, $identifier)
            ->where('is_active', true)
            ->firstOrFail();

        // Try to find existing user for this employee or same email/phone
        $userQuery = User::where('employee_id', $employee->id);

        if (!empty($employee->email)) {
            $userQuery->orWhere('email', $employee->email);
        }
        if (!empty($employee->phone)) {
            $userQuery->orWhere('phone', $employee->phone);
        }

        $user = $userQuery->first();

        if (!$user) {
            // No user yet → create new employee user
            $user = User::create([
                'name'        => $employee->name,
                'email'       => $employee->email,
                'phone'       => $employee->phone,
                'role'        => 'employee',
                'employee_id' => $employee->id,
                'password'    => null, // OTP-only login
            ]);
        } else {
            // User exists (maybe created earlier) → sync data, link employee
            $user->name        = $employee->name;
            $user->employee_id = $employee->id;

            // If user is already admin, keep admin. Otherwise employee.
            if ($user->role !== 'admin') {
                $user->role = 'employee';
            }

            // Fill missing email/phone if empty
            if (!$user->email && $employee->email) {
                $user->email = $employee->email;
            }
            if (!$user->phone && $employee->phone) {
                $user->phone = $employee->phone;
            }

            $user->save();
        }

        Auth::login($user);

        session()->forget(['otp_identifier', 'otp_channel']);

        return redirect()->route('employee.dashboard');
    }

    public function logout()
    {
        Auth::logout();
        return redirect()->route('login');
    }
}
