<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\Expense;
use App\Models\ExpenseGroup;
use App\Models\Session as FinancialSession;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class ExpenseController extends Controller
{
    public function __construct()
    {
        // No route middleware here if you already apply 'auth' in routes
        // Keep ensureEmployee() for role enforcement
    }

    protected function ensureEmployee()
    {
        $user = auth()->user();
        if (!$user || $user->role !== 'employee') {
            abort(403, 'Forbidden');
        }
        return $user;
    }

    /**
     * List expenses for logged-in employee
     */
    public function index(Request $request)
    {
        $user = $this->ensureEmployee();

        $expenses = Expense::where('user_id', $user->id)
            ->orderByDesc('created_at')
            ->paginate(12);

        return view('employee.expenses.index', compact('expenses'));
    }

    /**
     * Show create form (sheet style)
     */
    public function create()
    {
        $user = $this->ensureEmployee();

        $activeSession = FinancialSession::where('is_active', true)->first();
        $sessions = FinancialSession::orderByDesc('start_date')->get();

        // All active groups of this employee (user)
        $expenseGroups = ExpenseGroup::where('user_id', $user->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('employee.expenses.create', compact('activeSession', 'sessions', 'expenseGroups'));
    }

    /**
     * Store multiple expenses (rows[]) with multiple files per row.
     */
    public function store(Request $request)
    {
        $user = $this->ensureEmployee();

        $validator = Validator::make($request->all(), [
            'rows' => 'required|array|min:1',
            'rows.*.expense_date'      => 'required|date',
            'rows.*.expense_group_id'  => 'nullable|exists:expense_groups,id',
            'rows.*.group'             => 'nullable|string|max:191', // legacy/string if you still send
            'rows.*.party_name'        => 'nullable|string|max:191',
            'rows.*.purpose'           => 'nullable|string',
            'rows.*.amount'            => 'required|numeric|min:0',
            'rows.*.proof.*'           => 'nullable|file|mimes:png,jpg,jpeg,pdf|max:5120',
            'financial_session_id'     => 'nullable|exists:financial_sessions,id',
        ]);

        if ($validator->fails()) {
            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }
            return back()->withErrors($validator)->withInput();
        }

        // Decide session
        $sessionId = $request->input('financial_session_id');
        if (!$sessionId) {
            $active = FinancialSession::where('is_active', true)->first();
            if (!$active) {
                $msg = 'No active financial session found. Contact admin.';
                if ($request->wantsJson() || $request->ajax()) {
                    return response()->json(['message' => $msg], 422);
                }
                return back()->withErrors(['financial_session_id' => $msg]);
            }
            $sessionId = $active->id;
        }

        $createdIds = [];

        DB::beginTransaction();
        try {
            foreach ($request->input('rows') as $index => $row) {
                $expense = new Expense();
                $expense->user_id              = $user->id;
                $expense->employee_id          = $user->employee_id ?? null;
                $expense->financial_session_id = $sessionId;

                // ----- NEW: dynamic group handling -----
                $groupId   = $row['expense_group_id'] ?? null;
                $groupName = null;

                if ($groupId) {
                    $group = ExpenseGroup::where('id', $groupId)
                        ->where('user_id', $user->id) // security: must be this employee’s group
                        ->first();

                    if ($group) {
                        $groupName = $group->name;
                    } else {
                        // invalid / not owned → ignore id
                        $groupId = null;
                    }
                }

                $expense->expense_group_id = $groupId;
                $expense->group            = $groupName ?? ($row['group'] ?? null); // fallback to text if you still send it
                // ---------------------------------------

                $expense->expense_date = $row['expense_date'];
                $expense->party_name   = $row['party_name'] ?? null;
                $expense->purpose      = $row['purpose'] ?? null;
                $expense->amount       = $row['amount'];

                // Save early to get ID for folder naming
                $expense->save();

                // Files for this row: input name is rows.{i}.proof (array)
                $storedFiles = [];
                $files = $request->file("rows.{$index}.proof");
                if (is_array($files)) {
                    foreach ($files as $file) {
                        if (!$file || !$file->isValid()) continue;
                        $original = $file->getClientOriginalName();
                        $filename = now()->format('Ymd_His') . '_' . Str::random(6) . '.' . $file->getClientOriginalExtension();
                        // store to public disk under per-user/per-expense folder
                        $path = $file->storeAs("expenses/{$user->id}/{$expense->id}", $filename, 'public');
                        $storedFiles[] = [
                            'path'          => $path,
                            'original_name' => $original,
                        ];
                    }
                }

                // Persist proof(s)
                if (!empty($storedFiles) && Schema::hasColumn('expenses', 'proof_paths')) {
                    $expense->proof_paths = $storedFiles; // model should cast to array
                    $expense->save();
                } elseif (!empty($storedFiles)) {
                    // fallback: store first file into existing single-file columns
                    $expense->proof_path          = $storedFiles[0]['path'];
                    $expense->proof_original_name = $storedFiles[0]['original_name'];
                    $expense->save();
                }

                $createdIds[] = $expense->id;
            }

            DB::commit();

            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['message' => 'Expenses submitted successfully.', 'created' => $createdIds], 200);
            }
            return redirect()->route('employee.expenses.index')->with('success', 'Expenses submitted successfully.');

        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('Expense store error: '.$e->getMessage(), ['exception' => $e]);
            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['message' => 'Server error while saving expenses.'], 500);
            }
            return back()->withErrors(['server' => 'Server error while saving expenses.'])->withInput();
        }
    }

    /**
     * Show a single expense (employee-owned)
     */
    public function show(Expense $expense)
    {
        $user = $this->ensureEmployee();
        if ($expense->user_id !== $user->id) abort(403);

        return view('employee.expenses.show', compact('expense'));
    }

    /**
     * Edit single expense (employee)
     */
    public function edit(Expense $expense)
    {
        $user = $this->ensureEmployee();
        if ($expense->user_id !== $user->id) abort(403);

        // Load this employee's groups for dropdown on edit page
        $expenseGroups = ExpenseGroup::where('user_id', $user->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('employee.expenses.edit', compact('expense', 'expenseGroups'));
    }

    /**
     * Update single expense
     */
    public function update(Request $request, Expense $expense)
    {
        $user = $this->ensureEmployee();
        if ($expense->user_id !== $user->id) abort(403);

        $rules = [
            'expense_group_id' => 'nullable|exists:expense_groups,id',
            'group'            => 'nullable|string|max:191', // legacy string
            'expense_date'     => 'required|date',
            'party_name'       => 'nullable|string|max:191',
            'purpose'          => 'nullable|string',
            'amount'           => 'required|numeric|min:0',
            'add_files.*'      => 'nullable|file|mimes:png,jpg,jpeg,pdf|max:5120',
            'remove_files'     => 'nullable|array',
            'remove_files.*'   => 'string', // expect stored path strings
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }
            return back()->withErrors($validator)->withInput();
        }

        DB::beginTransaction();
        try {
            // ----- NEW: group update -----
            $groupId   = $request->input('expense_group_id');
            $groupName = null;

            if ($groupId) {
                $group = ExpenseGroup::where('id', $groupId)
                    ->where('user_id', $user->id)
                    ->first();

                if ($group) {
                    $groupName = $group->name;
                } else {
                    $groupId = null;
                }
            }

            $expense->expense_group_id = $groupId;
            $expense->group            = $groupName ?? $request->input('group');
            // ------------------------------------

            $expense->expense_date = $request->input('expense_date');
            $expense->party_name   = $request->input('party_name');
            $expense->purpose      = $request->input('purpose');
            $expense->amount       = $request->input('amount');
            $expense->save();

            // Remove any requested files (only if stored path belongs to this expense)
            if ($request->filled('remove_files') && is_array($request->input('remove_files'))) {
                $remove = $request->input('remove_files', []);
                $currentPaths = [];

                if (Schema::hasColumn('expenses', 'proof_paths') && $expense->proof_paths) {
                    $currentPaths = $expense->proof_paths; // array of ['path'=>..,'original_name'=>..]
                } else {
                    if ($expense->proof_path) {
                        $currentPaths = [['path' => $expense->proof_path, 'original_name' => $expense->proof_original_name]];
                    }
                }

                $remaining = [];
                foreach ($currentPaths as $item) {
                    if (in_array($item['path'], $remove, true)) {
                        // delete file from disk
                        if (Storage::disk('public')->exists($item['path'])) {
                            Storage::disk('public')->delete($item['path']);
                        }
                    } else {
                        $remaining[] = $item;
                    }
                }

                // Persist remaining
                if (Schema::hasColumn('expenses', 'proof_paths')) {
                    $expense->proof_paths = $remaining;
                    $expense->save();
                } else {
                    if (!empty($remaining)) {
                        $expense->proof_path          = $remaining[0]['path'];
                        $expense->proof_original_name = $remaining[0]['original_name'];
                    } else {
                        $expense->proof_path          = null;
                        $expense->proof_original_name = null;
                    }
                    $expense->save();
                }
            }

            // Add new files if any (add_files[] input)
            $addedFiles = [];
            $files = $request->file('add_files');
            if (is_array($files)) {
                foreach ($files as $file) {
                    if (!$file || !$file->isValid()) continue;
                    $original = $file->getClientOriginalName();
                    $filename = now()->format('Ymd_His') . '_' . Str::random(6) . '.' . $file->getClientOriginalExtension();
                    $path = $file->storeAs("expenses/{$user->id}/{$expense->id}", $filename, 'public');
                    $addedFiles[] = ['path' => $path, 'original_name' => $original];
                }
            }

            // Merge added files into existing list
            if (!empty($addedFiles)) {
                if (Schema::hasColumn('expenses', 'proof_paths')) {
                    $existing = $expense->proof_paths ?? [];
                    $expense->proof_paths = array_merge($existing, $addedFiles);
                    $expense->save();
                } else {
                    // If fallback single file columns, keep first existing or replace
                    if (!empty($addedFiles)) {
                        $expense->proof_path          = $addedFiles[0]['path'];
                        $expense->proof_original_name = $addedFiles[0]['original_name'];
                        $expense->save();
                    }
                }
            }

            DB::commit();

            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['message' => 'Expense updated.'], 200);
            }
            return redirect()->route('employee.expenses.index')->with('success', 'Expense updated.');

        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('Expense update error: '.$e->getMessage(), ['exception' => $e]);
            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['message' => 'Server error while updating expense.'], 500);
            }
            return back()->withErrors(['server' => 'Server error while updating expense.'])->withInput();
        }
    }

    /**
     * Delete expense and all associated files
     */
    public function destroy(Expense $expense, Request $request)
    {
        $user = $this->ensureEmployee();
        if ($expense->user_id !== $user->id) abort(403);

        DB::beginTransaction();
        try {
            // delete files on disk
            if (Schema::hasColumn('expenses', 'proof_paths') && $expense->proof_paths) {
                foreach ($expense->proof_paths as $file) {
                    if (!empty($file['path']) && Storage::disk('public')->exists($file['path'])) {
                        Storage::disk('public')->delete($file['path']);
                    }
                }
                // also delete parent folder if exists
                Storage::disk('public')->deleteDirectory("expenses/{$user->id}/{$expense->id}");
            } else {
                if ($expense->proof_path && Storage::disk('public')->exists($expense->proof_path)) {
                    Storage::disk('public')->delete($expense->proof_path);
                    Storage::disk('public')->deleteDirectory("expenses/{$user->id}/{$expense->id}");
                }
            }

            $expense->delete();
            DB::commit();

            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['message' => 'Expense deleted.'], 200);
            }
            return redirect()->route('employee.expenses.index')->with('success', 'Expense deleted.');

        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('Expense delete error: '.$e->getMessage(), ['exception' => $e]);
            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['message' => 'Server error while deleting expense.'], 500);
            }
            return back()->withErrors(['server' => 'Server error while deleting expense.']);
        }
    }

    /**
     * Download a proof file.
     * If proof_paths exists, pass query param ?file=<path> (urlencoded) OR ?index=N
     * Otherwise it will serve proof_path.
     */
    

public function downloadProof(Request $request, Expense $expense)
{
    $file = $request->query('file');

    if (!$file) {
        abort(404, "File not specified.");
    }

    // proof_paths array handling
    $availableFiles = [];

    if (is_array($expense->proof_paths)) {
        foreach ($expense->proof_paths as $pf) {
            $availableFiles[] = $pf['path'];
        }
    } elseif ($expense->proof_path) {
        $availableFiles[] = $expense->proof_path;
    }

    // security check
    if (!in_array($file, $availableFiles)) {
        abort(404, "This attachment does not belong to this expense.");
    }

    // remove leading slash if exists
    $file = ltrim($file, "/");

    // check storage
    if (!Storage::disk('public')->exists($file)) {
        abort(404, "File not available on server.");
    }

    return Storage::disk('public')->download($file);
}

   

}
