@extends('layouts.admin')

@section('page-title', 'Expense Report')
@section('page-subtitle', 'Filter, review, and export employee expenses')

@section('content')
    @php
        use Illuminate\Support\Str;
    @endphp

    <div class="space-y-6">

        {{-- FILTER CARD --}}
        <div class="bg-[color:var(--surface)] border border-[color:var(--border)] rounded-3xl shadow-sm p-4 md:p-5">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-3 mb-4">
                <div>
                    <h2 class="text-lg md:text-xl font-semibold">Filters</h2>
                    <p class="text-xs md:text-sm text-[color:var(--muted)] mt-1">
                        Choose a month or custom date range. You can also filter by employee.
                    </p>
                </div>

                {{-- Export button --}}
                <div class="flex gap-2">
                    <a href="{{ route('admin.expenses.report.export', request()->query()) }}"
                        class="inline-flex items-center gap-2 px-3 py-2 rounded-xl bg-emerald-600 text-white text-xs md:text-sm font-semibold hover:bg-emerald-700 transition shadow-sm">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M4 4v16h16V4H4zm4 5h8M8 12h5m-5 3h3" />
                        </svg>
                        Export to Excel
                    </a>
                </div>
            </div>

            <form method="GET" action="{{ route('admin.expenses.report') }}"
                class="space-y-3 md:space-y-0 md:grid md:grid-cols-8 md:gap-3">
                {{-- Month --}}
                <div class="md:col-span-2">
                    <label class="block text-xs font-medium text-[color:var(--muted)] mb-1">
                        Month (quick filter)
                    </label>
                    <input type="month" name="month" value="{{ $month ?? '' }}"
                        class="w-full border border-[color:var(--border)] rounded-xl px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-[color:var(--brand)] bg-white">
                    <p class="text-[10px] text-[color:var(--muted)] mt-1">
                        If selected, from/to will auto-fill for that month.
                    </p>
                </div>

                {{-- From Date --}}
                <div class="md:col-span-2">
                    <label class="block text-xs font-medium text-[color:var(--muted)] mb-1">
                        From date
                    </label>
                    <input type="date" name="from_date" value="{{ $fromDate ?? '' }}"
                        class="w-full border border-[color:var(--border)] rounded-xl px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-[color:var(--brand)] bg-white">
                </div>

                {{-- To Date --}}
                <div class="md:col-span-2">
                    <label class="block text-xs font-medium text-[color:var(--muted)] mb-1">
                        To date
                    </label>
                    <input type="date" name="to_date" value="{{ $toDate ?? '' }}"
                        class="w-full border border-[color:var(--border)] rounded-xl px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-[color:var(--brand)] bg-white">
                </div>

                {{-- Employee --}}
                <div class="md:col-span-2">
                    <label class="block text-xs font-medium text-[color:var(--muted)] mb-1">
                        Employee
                    </label>
                    <select name="employee_id"
                        class="w-full border border-[color:var(--border)] rounded-xl px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-[color:var(--brand)] bg-white">
                        <option value="">All employees</option>
                        @foreach($employees as $emp)
                            <option value="{{ $emp->id }}" {{ (int) $employeeId === $emp->id ? 'selected' : '' }}>
                                {{ $emp->name ?? $emp->email }}
                            </option>
                        @endforeach
                    </select>
                </div>

                <div class="md:col-span-8 flex flex-wrap gap-2 justify-end pt-2">
                    <a href="{{ route('admin.expenses.report') }}"
                        class="px-3 py-2 rounded-xl border border-[color:var(--border)] text-xs md:text-sm text-[color:var(--muted)] hover:bg-slate-50">
                        Reset
                    </a>
                    <button type="submit"
                        class="px-4 py-2 rounded-xl bg-[color:var(--brand)] text-white text-xs md:text-sm font-semibold hover:bg-[color:var(--brand-dark)] shadow-sm">
                        Apply Filters
                    </button>
                </div>
            </form>
        </div>

        {{-- SUMMARY STRIP --}}
        <div class="grid grid-cols-1 md:grid-cols-4 gap-3">
            <div class="bg-[color:var(--surface)] border border-[color:var(--border)] rounded-2xl p-3">
                <div class="text-[10px] uppercase tracking-[0.14em] text-[color:var(--muted)] mb-1">Total Records</div>
                <div class="text-xl font-bold">{{ $totalRecords ?? 0 }}</div>
            </div>
            <div class="bg-[color:var(--surface)] border border-[color:var(--border)] rounded-2xl p-3">
                <div class="text-[10px] uppercase tracking-[0.14em] text-[color:var(--muted)] mb-1">Total Amount</div>
                <div class="text-xl font-bold">
                    ₹ {{ number_format($totalAmount ?? 0, 2) }}
                </div>
            </div>
            <div class="bg-[color:var(--surface)] border border-[color:var(--border)] rounded-2xl p-3">
                <div class="text-[10px] uppercase tracking-[0.14em] text-[color:var(--muted)] mb-1">Active Session</div>
                <div class="text-sm font-semibold">
                    {{ $activeSession->name ?? 'Not Set' }}
                </div>
            </div>
            <div class="bg-[color:var(--surface)] border border-[color:var(--border)] rounded-2xl p-3">
                <div class="text-[10px] uppercase tracking-[0.14em] text-[color:var(--muted)] mb-1">Current Range</div>
                <div class="text-xs text-[color:var(--muted)]">
                    @if($fromDate || $toDate)
                        {{ $fromDate ?? '...' }} – {{ $toDate ?? '...' }}
                    @else
                        All dates
                    @endif
                </div>
            </div>
        </div>

        @if($mode === 'summary')
            {{-- SUMMARY TABLE --}}
            <div class="bg-[color:var(--surface)] border border-[color:var(--border)] rounded-3xl shadow-sm overflow-hidden">
                <div class="border-b border-[color:var(--border)] px-4 py-3 flex items-center justify-between">
                    <div class="text-sm font-semibold">Employee Summary</div>
                    <div class="text-xs text-[color:var(--muted)]">
                        Showing top employees by expense
                    </div>
                </div>

                <div class="overflow-x-auto">
                    <table class="min-w-full text-xs md:text-sm">
                        <thead class="bg-[color:var(--bg-2)] text-[color:var(--muted)] uppercase text-[10px]">
                            <tr>
                                <th class="px-4 py-3 text-left">Employee Name</th>
                                <th class="px-4 py-3 text-left">Emp Code</th>
                                <th class="px-4 py-3 text-right">No. of Claims</th>
                                <th class="px-4 py-3 text-right">Total Amount (₹)</th>
                                <th class="px-4 py-3 text-center">Action</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-[color:var(--border)]">
                            @forelse($summaryData as $row)
                                <tr class="hover:bg-slate-50/70 transition">
                                    <td class="px-4 py-3 font-medium text-gray-900">
                                        {{ $row->name }}
                                    </td>
                                    <td class="px-4 py-3 text-gray-500">
                                        {{ $row->employee->emp_code ?? '-' }}
                                    </td>
                                    <td class="px-4 py-3 text-right font-medium">
                                        {{ $row->expenses_count }}
                                    </td>
                                    <td class="px-4 py-3 text-right font-bold text-emerald-700">
                                        ₹ {{ number_format($row->expenses_sum_amount, 2) }}
                                    </td>
                                    <td class="px-4 py-3 text-center">
                                        <a href="{{ route('admin.expenses.report', array_merge(request()->query(), ['employee_id' => $row->id])) }}"
                                            class="inline-flex items-center gap-1 px-3 py-1.5 rounded-lg bg-indigo-50 text-indigo-700 hover:bg-indigo-100 text-xs font-semibold transition">
                                            View Details
                                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M9 5l7 7-7 7" />
                                            </svg>
                                        </a>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="5" class="px-6 py-8 text-center text-gray-500">
                                        No expenses found for this period.
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>

        @else
            {{-- DETAIL TABLE --}}
            <div class="bg-[color:var(--surface)] border border-[color:var(--border)] rounded-3xl shadow-sm overflow-hidden">
                <div class="border-b border-[color:var(--border)] px-4 py-3 flex items-center justify-between bg-indigo-50/50">
                    <div class="flex items-center gap-2">
                        <a href="{{ route('admin.expenses.report', \Illuminate\Support\Arr::except(request()->query(), ['employee_id'])) }}"
                            class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-white border border-gray-200 text-gray-500 hover:text-indigo-600 hover:border-indigo-200 transition"
                            title="Back to Summary">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                            </svg>
                        </a>
                        <div>
                            <div class="text-sm font-bold text-gray-900">{{ $selectedEmployee->name ?? 'Employee' }}</div>
                            <div class="text-[10px] text-gray-500 uppercase tracking-wide">Detailed Expense Report</div>
                        </div>
                    </div>
                    <div class="text-xs text-[color:var(--muted)]">
                        {{ $expenses->total() }} records found
                    </div>
                </div>

                <div class="overflow-x-auto">
                    <table class="min-w-full text-xs md:text-sm">
                        <thead class="bg-[color:var(--bg-2)] text-[color:var(--muted)] uppercase text-[10px]">
                            <tr>
                                <th class="px-3 py-2 text-left">Date</th>
                                <th class="px-3 py-2 text-left">Group</th>
                                <th class="px-3 py-2 text-left">Party</th>
                                <th class="px-3 py-2 text-left">Purpose</th>
                                <th class="px-3 py-2 text-right">Amount (₹)</th>
                                <th class="px-3 py-2 text-left">Session</th>
                                <th class="px-3 py-2 text-left">Created</th>
                                <th class="px-3 py-2 text-center">Attachments</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-[color:var(--border)]">
                            @forelse($expenses as $exp)
                                @php
                                    $files = [];
                                    if (!empty($exp->proof_paths) && is_array($exp->proof_paths)) {
                                        $files = $exp->proof_paths;
                                    } elseif (!empty($exp->proof_path)) {
                                        $files = [['path' => $exp->proof_path, 'original_name' => $exp->proof_original_name ?? basename($exp->proof_path)]];
                                    }
                                @endphp

                                <tr class="hover:bg-slate-50/70">
                                    <td class="px-3 py-2 align-top">
                                        <div class="font-medium text-[13px]">
                                            {{ $exp->expense_date?->format('d M Y') }}
                                        </div>
                                        <div class="text-[11px] text-[color:var(--muted)]">
                                            {{ $exp->created_at?->diffForHumans() }}
                                        </div>
                                    </td>
                                    <td class="px-3 py-2 align-top">
                                        <span
                                            class="inline-flex items-center px-2 py-0.5 rounded-full bg-[color:var(--brand-soft)] text-[color:var(--brand-dark)] text-[11px]">
                                            {{ $exp->group ?? 'General' }}
                                        </span>
                                    </td>
                                    <td class="px-3 py-2 align-top">
                                        {{ $exp->party_name ?? '-' }}
                                    </td>
                                    <td class="px-3 py-2 align-top">
                                        <div class="line-clamp-2">
                                            {{ $exp->purpose ?? '-' }}
                                        </div>
                                    </td>
                                    <td class="px-3 py-2 text-right align-top font-semibold">
                                        ₹ {{ number_format($exp->amount, 2) }}
                                    </td>
                                    <td class="px-3 py-2 align-top">
                                        {{ optional($exp->financialSession)->name ?? '-' }}
                                    </td>
                                    <td class="px-3 py-2 align-top text-[11px] text-[color:var(--muted)]">
                                        {{ $exp->created_at?->format('d M Y H:i') }}
                                    </td>
                                    <td class="px-3 py-2 align-top">
                                        @if(count($files))
                                            <div class="flex items-center gap-2">
                                                @foreach($files as $f)
                                                    @php
                                                        $filePath = $f['path'] ?? '';
                                                        $origName = $f['original_name'] ?? basename($filePath);
                                                        $downloadUrl = route('admin.expenses.download', $exp->id) . '?file=' . urlencode($filePath);
                                                    @endphp

                                                    <a href="{{ $downloadUrl }}" download="{{ $origName }}"
                                                        title="Download: {{ $origName }}"
                                                        class="inline-flex items-center justify-center w-8 h-8 rounded-md border border-[color:var(--border)] bg-white hover:bg-slate-50 text-[color:var(--muted)]">
                                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                                d="M4 16v2a2 2 0 002 2h12a2 2 0 002-2v-2M7 10l5 5 5-5M12 15V3" />
                                                        </svg>
                                                    </a>
                                                @endforeach
                                            </div>
                                        @else
                                            <div class="text-xs text-[color:var(--muted)]">—</div>
                                        @endif
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="8" class="px-3 py-6 text-center text-xs text-[color:var(--muted)]">
                                        No expenses found for this employee in selected period.
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                {{-- Pagination --}}
                @if($expenses->hasPages())
                    <div class="border-t border-[color:var(--border)] px-4 py-3">
                        {{ $expenses->links() }}
                    </div>
                @endif
            </div>
        @endif

        <style>
            table a[download] svg {
                color: inherit;
            }
        </style>
@endsection