@extends('layouts.employee')

@section('page-title', 'Edit Conveyance')
@section('page-subtitle', 'Edit your conveyance entry for the claim month')

@section('content')

    {{-- TOP SECTION --}}
    <div class="mb-6 flex flex-col md:flex-row md:items-center md:justify-between gap-4">
        <div>
            <h2 class="text-xl md:text-2xl font-extrabold tracking-tight">Edit Conveyance</h2>
            <p class="text-sm text-[color:var(--emp-muted)] mt-1">
                Use this sheet to update your selected conveyance row for
                <span class="font-semibold text-[color:var(--emp-text)]">{{ $targetMonthLabel }}</span>
                ({{ $monthStart->format('d M Y') }} – {{ $monthEnd->format('d M Y') }}).
            </p>
            <p class="text-xs text-[color:var(--emp-muted)] mt-1">
                Submission/edit allowed till {{ $cutoffDate->format('d M Y') }}.
            </p>
        </div>

        <div
            class="text-xs md:text-sm text-[color:var(--emp-muted)] bg-[color:var(--emp-surface)] border border-[color:var(--emp-border)] rounded-2xl px-4 py-3 max-w-md">
            <div class="font-semibold text-[11px] uppercase tracking-[0.16em] mb-1">Quick</div>
            <div class="space-y-1">
                <div>• Changes saved as <span class="font-semibold">Draft</span> remain editable.</div>
                <div>• Choose <span class="font-semibold">Submit (Final)</span> to send for approval.</div>
            </div>
        </div>
    </div>

    {{-- CARD --}}
    <div class="rounded-3xl bg-[color:var(--emp-surface)] border border-[color:var(--emp-border)] shadow-sm p-5 mb-6">
        {{-- Card header --}}
        <div class="mb-4 flex flex-col md:flex-row md:items-center md:justify-between gap-3">
            <div>
                <h3 class="text-lg md:text-xl font-semibold">Edit Entry — {{ $targetMonthLabel }}</h3>
                <p class="text-xs md:text-sm text-[color:var(--emp-muted)] mt-1">
                    Update fields below. Layout matches the Add Conveyance sheet for familiarity.
                </p>
            </div>

            <div
                class="text-xs md:text-sm text-[color:var(--emp-muted)] bg-[color:var(--emp-bg-soft)] border border-[color:var(--emp-border)] rounded-2xl px-3 py-2">
                <div class="font-semibold">Last editable date</div>
                <div>{{ $cutoffDate->format('d M Y') }}</div>
            </div>
        </div>

        {{-- CARD BANNER (in-card success / error) --}}
        <div id="cardBannerContainer" class="mb-4 hidden"></div>

        {{-- FORM --}}
        <form id="conveyanceForm" method="POST" action="{{ route('employee.conveyance.update', $conveyance) }}">
            @csrf
            @method('PUT')
            <input type="hidden" name="update_mode" id="updateModeInput" value="draft">

            <div class="rounded-2xl border border-[color:var(--emp-border)] bg-white overflow-hidden">

                <div class="max-h-[420px] overflow-y-auto">
                    <table class="w-full text-[13px] table-fixed responsive-table">
                        <thead
                            class="sticky top-0 z-10 bg-slate-50 text-[11px] uppercase tracking-wide text-[color:var(--emp-muted)]">
                            <tr>
                                <th class="p-2 md:p-3 text-left font-semibold w-[7rem]">Date</th>
                                <th class="p-2 md:p-3 text-left font-semibold w-[8rem]">From</th>
                                <th class="p-2 md:p-3 text-left font-semibold w-[8rem]">To</th>
                                <th class="p-2 md:p-3 text-left font-semibold w-[10rem]">Client</th>
                                <th class="p-2 md:p-3 text-left font-semibold w-[8rem]">Mode</th>
                                <th class="p-2 md:p-3 text-right font-semibold w-[7rem]">Amount (₹)</th>
                                <th class="p-2 md:p-3 text-left font-semibold">Remarks</th>
                                <th class="p-2 md:p-3 text-center font-semibold w-[4rem]">Return</th>
                                <th class="p-2 md:p-3 text-center font-semibold w-[3rem]"></th>
                            </tr>
                        </thead>
                        <tbody id="conv-rows" class="divide-y divide-[color:var(--emp-border)]">
                            {{-- Single editable row (mirrors create's row structure) --}}
                            <tr data-row-type="main" data-row-id="edit-row-1" class="hover:bg-slate-50 transition">
                                <td data-label="Date" class="p-2 md:p-3 align-top">
                                    <div class="md:hidden mobile-label">Date</div>
                                    <div class="field-wrap">
                                        <input type="date" name="date" id="field_date"
                                            min="{{ $monthStart->format('Y-m-d') }}" max="{{ $monthEnd->format('Y-m-d') }}"
                                            value="{{ old('date', $conveyance->date->format('Y-m-d')) }}"
                                            class="w-full border border-[color:var(--emp-border)] rounded-xl text-[13px] focus:outline-none focus:ring-2 focus:ring-[color:var(--emp-brand)]" />
                                    </div>
                                </td>

                                <td data-label="From" class="p-2 md:p-3 align-top">
                                    <div class="md:hidden mobile-label">From</div>
                                    <div class="field-wrap">
                                        <input type="text" name="from" id="field_from" data-field="from"
                                            value="{{ old('from', $conveyance->from_location) }}"
                                            class="w-full border border-[color:var(--emp-border)] rounded-xl text-[13px] focus:outline-none focus:ring-2 focus:ring-[color:var(--emp-brand)]"
                                            placeholder="From">
                                    </div>
                                </td>

                                <td data-label="To" class="p-2 md:p-3 align-top">
                                    <div class="md:hidden mobile-label">To</div>
                                    <div class="field-wrap">
                                        <input type="text" name="to" id="field_to" data-field="to"
                                            value="{{ old('to', $conveyance->to_location) }}"
                                            class="w-full border border-[color:var(--emp-border)] rounded-xl text-[13px] focus:outline-none focus:ring-2 focus:ring-[color:var(--emp-brand)]"
                                            placeholder="To">
                                    </div>
                                </td>

                                <td data-label="Client" class="p-2 md:p-3 align-top">
                                    <div class="md:hidden mobile-label">Client</div>
                                    <div class="field-wrap">
                                        <select name="client_id" id="field_client" data-field="client" required
                                            class="w-full border border-[color:var(--emp-border)] rounded-xl text-[13px] bg-white focus:outline-none focus:ring-2 focus:ring-[color:var(--emp-brand)]">
                                            <option value="">Select client</option>
                                            @foreach($clients as $c)
                                                <option value="{{ $c->id }}" @selected(old('client_id', $conveyance->client_id) == $c->id)>{{ $c->name }}</option>
                                            @endforeach
                                            <option value="__request_new__">+ Request new client…</option>
                                        </select>
                                    </div>
                                </td>

                                <td data-label="Mode" class="p-2 md:p-3 align-top">
                                    <div class="md:hidden mobile-label">Mode</div>
                                    <div class="field-wrap">
                                        <input type="text" name="mode" id="field_mode" data-field="mode"
                                            value="{{ old('mode', $conveyance->mode) }}"
                                            class="w-full border border-[color:var(--emp-border)] rounded-xl text-[13px] focus:outline-none focus:ring-2 focus:ring-[color:var(--emp-brand)]"
                                            placeholder="e.g. Cab, Auto, Bus">
                                    </div>
                                </td>

                                <td data-label="Amount (₹)" class="p-2 md:p-3 text-right align-top">
                                    <div class="md:hidden mobile-label">Amount (₹)</div>
                                    <div class="field-wrap">
                                        <input type="number" step="0.01" min="0" name="amount" id="field_amount"
                                            data-field="amount"
                                            value="{{ old('amount', number_format($conveyance->amount, 2, '.', '')) }}"
                                            class="w-full border border-[color:var(--emp-border)] rounded-xl text-[13px] text-right focus:outline-none focus:ring-2 focus:ring-[color:var(--emp-brand)]"
                                            placeholder="0.00" oninput="recalcDisplayTotal()">
                                    </div>
                                </td>

                                <td data-label="Remarks" class="p-2 md:p-3 align-top">
                                    <div class="md:hidden mobile-label">Remarks</div>
                                    <div class="field-wrap">
                                        <input type="text" name="remarks" id="field_remarks"
                                            value="{{ old('remarks', $conveyance->remarks) }}"
                                            class="w-full border border-[color:var(--emp-border)] rounded-xl text-[13px] focus:outline-none focus:ring-2 focus:ring-[color:var(--emp-brand)]"
                                            placeholder="Optional remarks">
                                    </div>
                                </td>

                                <td data-label="Return" class="p-2 md:p-3 text-center align-top">
                                    <div class="md:hidden mobile-label">Return</div>
                                    <div class="field-wrap flex items-center justify-center">
                                        <input type="checkbox" name="is_return" id="field_is_return" value="1"
                                            data-field="return"
                                            class="h-4 w-4 rounded border-[color:var(--emp-border)] text-[color:var(--emp-brand)] focus:ring-[color:var(--emp-brand)] mr-2"
                                            @if(old('is_return', $conveyance->is_return)) checked @endif>
                                        <button type="button" id="editReturnBtn"
                                            class="px-2 py-1 rounded-lg text-xs border border-[color:var(--emp-border)] hover:bg-slate-50">Edit</button>
                                    </div>
                                </td>

                                <td data-label="Actions" class="p-2 md:p-3 text-center align-top">
                                    <div class="md:hidden mobile-label">Actions</div>
                                    <div class="field-wrap">
                                        <a href="{{ route('employee.conveyance.index') }}"
                                            class="text-[11px] px-2 py-1 rounded-lg bg-[color:var(--emp-bg-soft)] border border-[color:var(--emp-border)] hover:bg-slate-100 font-semibold">Back</a>
                                    </div>
                                    <!-- Hidden fields for return -->
                                    <input type="hidden" name="return_amount" id="field_return_amount"
                                        value="{{ old('return_amount', $conveyance->return_amount) }}">
                                    <input type="hidden" name="return_remarks" id="field_return_remarks"
                                        value="{{ old('return_remarks', $conveyance->return_remarks) }}">
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>

                {{-- TOTAL BAR --}}
                <div
                    class="border-t border-[color:var(--emp-border)] bg-slate-50 p-3 flex items-center justify-between gap-3">
                    <div class="text-[color:var(--emp-muted)] text-[11px]">
                        Current total (including return if enabled):
                    </div>
                    <div class="flex items-center gap-3">
                        <div id="displayTotal" class="font-extrabold text-[color:var(--emp-brand)] text-base">
                            {{-- initial display set by JS on DOMContentLoaded --}}
                            ₹ 0.00
                        </div>

                        <div class="flex gap-2">
                            <button type="button" id="saveDraftBtn"
                                class="px-4 py-2.5 rounded-xl bg-[color:var(--emp-bg-soft)] text-[color:var(--emp-text)] text-xs md:text-sm font-semibold border border-[color:var(--emp-border)] hover:bg-slate-100 transition">
                                Save Draft
                            </button>
                            <button type="submit" id="submitBtn"
                                class="px-4 py-2.5 rounded-xl bg-[color:var(--emp-brand)] text-white text-xs md:text-sm font-semibold hover:bg-blue-700 transition shadow-sm">
                                Submit (Final)
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>

    {{-- Client info + request box (same as create) --}}
    <div class="rounded-3xl bg-[color:var(--emp-surface)] border border-[color:var(--emp-border)] shadow-sm p-5 mb-6">
        <h4 class="text-sm font-semibold mb-2">Clients & Rules</h4>
        <ul class="text-xs text-[color:var(--emp-muted)] space-y-1.5">
            <li>• Client list below is coming from Admin (live data).</li>
            <li>• “Request new client” will submit a request to Admin (demo: simple flow).</li>
        </ul>

        <div id="clientRequestBox"
            class="mt-4 p-3 rounded-2xl bg-[color:var(--emp-bg-soft)] border border-[color:var(--emp-border)] hidden">
            <div class="flex items-center justify-between mb-2">
                <div class="text-xs font-semibold text-[color:var(--emp-text)]">Request new client</div>
                <button type="button" id="closeClientRequest"
                    class="text-[11px] text-[color:var(--emp-muted)] hover:text-[color:var(--emp-text)]">✕</button>
            </div>
            <div class="space-y-2">
                <div>
                    <label class="text-[11px] font-medium block mb-1">Client Name</label>
                    <input id="reqClientName"
                        class="w-full border border-[color:var(--emp-border)] rounded-xl px-3 py-2 text-xs focus:outline-none focus:ring-1 focus:ring-[color:var(--emp-brand)]"
                        placeholder="Proposed client name">
                </div>
                <div>
                    <label class="text-[11px] font-medium block mb-1">Details / Reason</label>
                    <textarea id="reqClientNotes" rows="2"
                        class="w-full border border-[color:var(--emp-border)] rounded-xl px-3 py-2 text-xs focus:outline-none focus:ring-1 focus:ring-[color:var(--emp-brand)]"
                        placeholder="Why do you need this client added?"></textarea>
                </div>
                <button type="button" id="fakeClientRequestBtn"
                    class="w-full inline-flex items-center justify-center gap-1 px-3 py-2 rounded-xl bg-[color:var(--emp-brand)] text-white text-xs font-semibold hover:bg-blue-700 transition">
                    Submit Request (UI only)
                </button>
                <div id="clientRequestMsg" class="text-[11px] text-emerald-600 mt-1 hidden">Request captured (demo only).
                </div>
            </div>
        </div>
    </div>

    {{-- Return Fare Modal (same UI as create) --}}
    <div id="returnModal" class="fixed inset-0 bg-black/40 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-2xl shadow-xl w-full max-w-sm mx-4">
            <div class="px-4 py-3 border-b border-slate-200 flex items-center justify-between">
                <h3 class="text-sm font-semibold text-slate-800">Return Trip Details</h3>
                <button type="button" id="returnModalClose"
                    class="text-xs text-slate-500 hover:text-slate-700">&times;</button>
            </div>

            <div class="p-4 space-y-4">
                {{-- Step 1: Ask if same fare --}}
                <div id="returnStepQuestion">
                    <p class="text-xs text-slate-600 mb-3">Is the fare for the return journey the same as the onward fare?
                    </p>
                    <div class="flex flex-col gap-2">
                        <button type="button" id="btnReturnSame"
                            class="w-full px-3 py-2 rounded-xl bg-[color:var(--emp-brand)] text-white text-xs font-semibold hover:bg-blue-700">Yes,
                            same fare (use ×2)</button>
                        <button type="button" id="btnReturnDifferent"
                            class="w-full px-3 py-2 rounded-xl border border-[color:var(--emp-border)] text-xs font-semibold text-[color:var(--emp-text)] hover:bg-slate-50">No,
                            different fare</button>
                    </div>
                </div>

                {{-- Step 2: Different fare inputs --}}
                <div id="returnStepDifferent" class="hidden space-y-3">
                    <div>
                        <label class="block text-[11px] font-medium mb-1">Return Amount (₹)</label>
                        <input type="number" step="0.01" min="0" id="returnDifferentAmount"
                            class="w-full border border-[color:var(--emp-border)] rounded-xl px-3 py-2 text-xs focus:outline-none focus:ring-1 focus:ring-[color:var(--emp-brand)]"
                            placeholder="Enter return fare">
                    </div>
                    <div>
                        <label class="block text-[11px] font-medium mb-1">Return Remark (optional)</label>
                        <input type="text" id="returnDifferentRemark"
                            class="w-full border border-[color:var(--emp-border)] rounded-xl px-3 py-2 text-xs focus:outline-none focus:ring-1 focus:ring-[color:var(--emp-brand)]"
                            placeholder="e.g. different route, surge pricing">
                    </div>
                    <div class="flex justify-end gap-2 pt-1">
                        <button type="button" id="btnReturnDifferentBack"
                            class="px-3 py-1.5 rounded-xl text-[11px] border border-[color:var(--emp-border)] text-[color:var(--emp-muted)] hover:bg-slate-50">Back</button>
                        <button type="button" id="btnReturnDifferentSave"
                            class="px-3 py-1.5 rounded-xl bg-[color:var(--emp-brand)] text-white text-[11px] font-semibold hover:bg-blue-700">Save
                            return fare</button>
                    </div>
                </div>
            </div>

            <div class="px-4 py-3 border-t border-slate-200 flex justify-end">
                <button type="button" id="returnModalCancel"
                    class="px-3 py-1.5 rounded-xl text-[11px] text-[color:var(--emp-muted)] hover:bg-slate-50">Cancel
                    return</button>
            </div>
        </div>
    </div>

    @php
        $clientOptionsHtml = '<option value=\"\">Select client</option>';
        foreach ($clients as $c) {
            $clientOptionsHtml .= '<option value=\"' . $c->id . '\">' . e($c->name) . '</option>';
        }
        $clientOptionsHtml .= '<option value=\"__request_new__\">+ Request new client…</option>';
    @endphp

    {{-- Styles: equal input heights + mobile labels above inputs --}}
    <style>
        /* Ensure consistent sizing of inputs/selects across browsers */
        .responsive-table input[type="text"],
        .responsive-table input[type="number"],
        .responsive-table input[type="date"],
        .responsive-table select,
        .responsive-table textarea {
            box-sizing: border-box;
            height: 44px;
            /* consistent height */
            padding-top: 0.35rem;
            padding-bottom: 0.35rem;
            padding-left: 0.6rem;
            padding-right: 0.6rem;
            line-height: 1.2;
        }

        .responsive-table textarea {
            height: 44px;
            min-height: 44px;
            padding-top: 0.35rem;
            padding-bottom: 0.35rem;
        }

        /* Checkbox small remains small */
        .responsive-table input[type="checkbox"] {
            height: auto;
            width: auto;
        }

        /* Date picker indicator visual consistency */
        .responsive-table input[type="date"]::-webkit-calendar-picker-indicator {
            opacity: 0.9;
        }

        /* Responsive table: on small screens convert rows to labelled blocks */
        @media (max-width: 767px) {
            .responsive-table thead {
                display: none;
            }

            .responsive-table,
            .responsive-table tbody,
            .responsive-table tr,
            .responsive-table td {
                display: block;
                width: 100%;
            }

            .responsive-table tr {
                margin-bottom: 0.75rem;
                padding: 0.75rem;
                border-radius: 0.75rem;
                background: white;
                box-shadow: 0 1px 0 rgba(0, 0, 0, 0.03);
            }

            .responsive-table td {
                padding: 0.35rem 0;
                display: block;
                gap: 0.5rem;
                border-bottom: 0;
            }

            /* Hide pseudo label (we're using explicit label elements for better control) */
            .responsive-table td::before {
                display: none !important;
            }

            /* Mobile-only label (inserted above input) */
            .mobile-label {
                display: block;
                font-size: 11px;
                color: rgba(30, 41, 59, 0.6);
                font-weight: 600;
                margin-bottom: 0.35rem;
            }

            .responsive-table td .field-wrap {
                display: block;
                width: 100%;
            }

            .responsive-table td .field-wrap input,
            .responsive-table td .field-wrap select,
            .responsive-table td .field-wrap textarea {
                width: 100%;
            }

            .responsive-table td.actions {
                display: flex;
                justify-content: flex-end;
            }
        }

        /* Card banner helper */
        .card-banner-success {
            background-color: #ecfdf5;
            /* emerald-50 */
            border: 1px solid #bbf7d0;
            /* emerald-200 */
            color: #065f46;
            /* emerald-800 */
            padding: 0.6rem 0.9rem;
            border-radius: 0.75rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-weight: 600;
        }

        .card-banner-error {
            background-color: #fff1f2;
            /* rose-50 */
            border: 1px solid #fecaca;
            /* rose-200 */
            color: #9f1239;
            /* rose-800 */
            padding: 0.6rem 0.9rem;
            border-radius: 0.75rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-weight: 600;
        }
    </style>

    {{-- Optional floating global banner kept for dev/debug (not used for submit messages now) --}}
    <div id="globalBanner"
        class="fixed inset-x-0 top-4 z-60 pointer-events-none transition-transform transform -translate-y-6 opacity-0"
        aria-live="polite" role="status">
        <div id="globalBannerInner"
            class="mx-auto max-w-3xl rounded-2xl px-4 py-3 shadow-lg text-sm font-medium flex items-start gap-3 pointer-events-auto">
            <!-- dynamic content inserted here -->
        </div>
    </div>

    <script>
        const clientOptionsHtml = "{!! $clientOptionsHtml !!}";
        const minDate = "{{ $monthStart->format('Y-m-d') }}";
        const maxDate = "{{ $monthEnd->format('Y-m-d') }}";
        const clientRequestUrl = "{{ route('employee.client-requests.store') }}";

        // DOM refs
        const rowsContainer = document.getElementById('conv-rows');
        const totalAmountCell = document.getElementById('displayTotal');
        const cardBannerContainer = document.getElementById('cardBannerContainer');

        // inputs
        const amountInput = document.getElementById('field_amount');
        const isReturnCheckbox = document.getElementById('field_is_return');
        const returnAmountInput = document.getElementById('field_return_amount');
        const returnRemarkInput = document.getElementById('field_return_remarks');

        // return modal elements
        let returnModal = document.getElementById('returnModal');
        let returnStepQuestion = document.getElementById('returnStepQuestion');
        let returnStepDifferent = document.getElementById('returnStepDifferent');
        let btnReturnSame = document.getElementById('btnReturnSame');
        let btnReturnDifferent = document.getElementById('btnReturnDifferent');
        let btnReturnDifferentBack = document.getElementById('btnReturnDifferentBack');
        let btnReturnDifferentSave = document.getElementById('btnReturnDifferentSave');
        let btnReturnModalClose = document.getElementById('returnModalClose');
        let btnReturnModalCancel = document.getElementById('returnModalCancel');
        let inputReturnDifferentAmount = document.getElementById('returnDifferentAmount');
        let inputReturnDifferentRemark = document.getElementById('returnDifferentRemark');

        // client request box
        const clientRequestBox = document.getElementById('clientRequestBox');

        function formatCurrency(value) {
            const num = isNaN(value) ? 0 : Number(value);
            return '₹ ' + num.toFixed(2);
        }

        // Card banner helper
        let cardBannerTimeout = null;
        function showCardBanner(message, type = 'success', persist = false) {
            cardBannerContainer.innerHTML = '';
            cardBannerContainer.classList.remove('hidden');

            const div = document.createElement('div');
            div.className = (type === 'error') ? 'card-banner-error' : 'card-banner-success';
            div.setAttribute('role', 'status');
            div.innerHTML = `
                <div class="flex items-center gap-2">
                    <div aria-hidden="true">
                        ${type === 'error'
                    ? '<svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-width="2" stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"/></svg>'
                    : '<svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-width="2" stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7" /></svg>'
                }
                    </div>
                    <div>${message}</div>
                </div>
                <div class="ml-auto">
                    <button id="dismissCardBanner" class="text-xs underline">Dismiss</button>
                </div>
            `;
            cardBannerContainer.appendChild(div);

            document.getElementById('dismissCardBanner')?.addEventListener('click', hideCardBanner);

            if (!persist) {
                if (cardBannerTimeout) clearTimeout(cardBannerTimeout);
                cardBannerTimeout = setTimeout(hideCardBanner, 5000);
            }
        }

        function hideCardBanner() {
            cardBannerContainer.classList.add('hidden');
            cardBannerContainer.innerHTML = '';
            if (cardBannerTimeout) {
                clearTimeout(cardBannerTimeout);
                cardBannerTimeout = null;
            }
        }

        // Initialize display total
        function recalcDisplayTotal() {
            const a = parseFloat(amountInput?.value || '0') || 0;
            const ret = parseFloat(returnAmountInput?.value || '0');
            let total = a;
            if (isReturnCheckbox?.checked) {
                if (!isNaN(ret) && ret > 0) total = a + ret;
                else total = a * 2;
            }
            totalAmountCell.textContent = formatCurrency(total);
        }

        document.addEventListener('DOMContentLoaded', () => {
            // initial total
            recalcDisplayTotal();

            // amount change
            amountInput?.addEventListener('input', recalcDisplayTotal);

            // is_return toggle
            isReturnCheckbox?.addEventListener('change', () => {
                if (isReturnCheckbox.checked) {
                    const clientVal = document.getElementById('field_client')?.value;
                    if (!clientVal) {
                        showCardBanner('Please select a client before adding a return trip.', 'error');
                        isReturnCheckbox.checked = false;
                        document.getElementById('field_client')?.focus();
                        return; // stop here
                    }
                }

                if (!isReturnCheckbox.checked) {
                    // clear return fields
                    returnAmountInput.value = '';
                    returnRemarkInput.value = '';
                }
                recalcDisplayTotal();
            });

            // Return modal open
            document.getElementById('editReturnBtn')?.addEventListener('click', () => {
                // populate modal from hidden values
                inputReturnDifferentAmount.value = returnAmountInput.value || '';
                inputReturnDifferentRemark.value = returnRemarkInput.value || '';
                returnStepQuestion.classList.remove('hidden');
                returnStepDifferent.classList.add('hidden');
                returnModal.classList.remove('hidden');
                inputReturnDifferentAmount.focus();
            });

            // Modal actions
            btnReturnSame?.addEventListener('click', () => {
                // same fare -> clear custom return amount
                returnAmountInput.value = '';
                returnRemarkInput.value = '';
                isReturnCheckbox.checked = true;
                recalcDisplayTotal();
                closeReturnModal();
            });

            btnReturnDifferent?.addEventListener('click', () => {
                returnStepQuestion.classList.add('hidden');
                returnStepDifferent.classList.remove('hidden');
                inputReturnDifferentAmount.focus();
            });

            btnReturnDifferentBack?.addEventListener('click', () => {
                returnStepDifferent.classList.add('hidden');
                returnStepQuestion.classList.remove('hidden');
            });

            btnReturnDifferentSave?.addEventListener('click', () => {
                const val = parseFloat(inputReturnDifferentAmount.value || '0');
                const remark = inputReturnDifferentRemark.value.trim();
                if (isNaN(val) || val <= 0) {
                    alert('Please enter a valid return amount greater than 0.');
                    inputReturnDifferentAmount.focus();
                    return;
                }
                returnAmountInput.value = val.toFixed(2);
                returnRemarkInput.value = remark;
                isReturnCheckbox.checked = true;
                recalcDisplayTotal();
                closeReturnModal();
            });

            btnReturnModalClose?.addEventListener('click', () => {
                // close and no changes
                closeReturnModal();
            });
            btnReturnModalCancel?.addEventListener('click', () => {
                // cancel and remove return flag
                // keep existing values intact (user may cancel)
                closeReturnModal();
            });

            // close if clicking backdrop
            if (returnModal) {
                returnModal.addEventListener('click', (e) => {
                    if (e.target === returnModal) closeReturnModal();
                });
            }

            function closeReturnModal() {
                if (!returnModal) return;
                returnModal.classList.add('hidden');
            }

            // Save Draft / Submit behaviour
            const saveDraftBtn = document.getElementById('saveDraftBtn');
            const submitBtn = document.getElementById('submitBtn');
            const updateModeInput = document.getElementById('updateModeInput');
            const form = document.getElementById('conveyanceForm');

            saveDraftBtn?.addEventListener('click', () => {
                updateModeInput.value = 'draft';
                form.submit();
            });

            submitBtn?.addEventListener('click', () => {
                updateModeInput.value = 'final';
            });

            // Client request flow (same as create)
            document.getElementById('field_client')?.addEventListener('change', function () {
                if (this.value === '__request_new__') {
                    this.value = '';
                    clientRequestBox.classList.remove('hidden');
                }
            });

            document.getElementById('closeClientRequest')?.addEventListener('click', () => {
                clientRequestBox.classList.add('hidden');
            });

            document.getElementById('fakeClientRequestBtn')?.addEventListener('click', async (e) => {
                const btn = e.currentTarget;
                const nameInput = document.getElementById('reqClientName');
                const notesInput = document.getElementById('reqClientNotes');

                const name = nameInput.value.trim();
                const notes = notesInput.value.trim();

                if (!name) {
                    showCardBanner('Please enter client name for request.', 'error');
                    nameInput.focus();
                    return;
                }

                const token = document.querySelector('meta[name="csrf-token"]').getAttribute('content');

                // Loading state
                btn.disabled = true;
                const originalHtml = btn.innerHTML;
                btn.innerHTML = `<svg class="animate-spin h-4 w-4 mr-2 inline-block" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8v4a4 4 0 00-4 4H4z"></path></svg>Sending...`;

                try {
                    const res = await fetch(clientRequestUrl, {
                        method: 'POST',
                        headers: {
                            'X-CSRF-TOKEN': token,
                            'Accept': 'application/json',
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({ client_name: name, notes: notes })
                    });

                    if (!res.ok) {
                        const data = await res.json().catch(() => ({}));
                        showCardBanner(data.message || 'Failed to submit client request.', 'error', true);
                        return;
                    }

                    const data = await res.json();
                    showCardBanner(data.message || 'Client request submitted to Admin.', 'success', true);

                    // Reset + hide
                    nameInput.value = '';
                    notesInput.value = '';
                    clientRequestBox.classList.add('hidden');
                } catch (err) {
                    console.error(err);
                    showCardBanner('Network error while submitting client request.', 'error', true);
                } finally {
                    btn.disabled = false;
                    btn.innerHTML = originalHtml;
                }
            });

            // Show server flash messages (if any)
            @if(session('success'))
                showCardBanner(@json(session('success')), 'success', true);
            @endif

            @if($errors->any())
                    showCardBanner(@json($errors->first()), 'error', true);
                    // highlight first invalid field if possible
                    @php
                        $first = collect($errors->keys())->first();
                    @endphp
                @if($first)
                    const fieldName = @json($first);
                    // Try to find matching element (simple heuristics)
                    setTimeout(() => {
                        const el = document.querySelector(`[name="${fieldName}"], [name="${fieldName.replace('.', '\\.')}"]`);
                        if (el) el.classList.add('border-red-400');
                        }, 120);
                @endif
            @endif
        });
    </script>

@endsection
