<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Route;
use Spatie\Sitemap\Sitemap; // If they have spatie, but better to be dependency-free for "Enterprise custom" if not sure.
// I will write a simple Native XML generator to avoid needing `composer require spatie/laravel-sitemap` which might fail if not allowed.
// Native XML generation is safer here.

class GenerateSitemap extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sitemap:generate';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate the sitemap.xml file';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Generating sitemap...');

        $routes = Route::getRoutes();
        $sitemapReqs = [];

        // 1. Static Routes (Home, etc)
        // We filter for GET routes that are not admin/auth
        foreach ($routes as $route) {
            if ($route->methods()[0] !== 'GET') continue;
            
            $uri = $route->uri();
            $name = $route->getName();
            
            // Exclude specific patterns
            if (
                str_starts_with($uri, 'admin') || 
                str_starts_with($uri, 'sanctum') || 
                str_starts_with($uri, '_') || 
                str_starts_with($uri, 'api') ||
                str_contains($uri, 'dashboard') ||
                str_contains($uri, 'login') ||
                str_contains($uri, 'register') ||
                str_contains($uri, 'password') ||
                str_contains($uri, '{') // Skip parameterized routes for auto-discovery (handle manually below)
            ) {
                continue;
            }

            $sitemapReqs[] = url($uri);
        }

        // 2. Dynamic Routes (Manual Addition)
        // e.g. Patient Stories, Updates. 
        // We'll fetch models here.
        
        // Patient Stories
        try {
            $stories = \App\Models\PatientStory::where('status', 'published')->get();
            foreach($stories as $story) {
                $sitemapReqs[] = route('patient-stories.show', $story->slug);
            }
        } catch (\Exception $e) {
            $this->warn('Could not fetch Patient Stories: ' . $e->getMessage());
        }

        // Updates
        try {
            $updates = \App\Models\LatestUpdate::all(); // Assuming 'published' check if exists
            foreach($updates as $update) {
                $sitemapReqs[] = route('updates.show', $update->slug);
            }
        } catch (\Exception $e) {
             $this->warn('Could not fetch Updates: ' . $e->getMessage());
        }


        // Build XML
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';

        foreach(array_unique($sitemapReqs) as $url) {
            $xml .= '<url>';
            $xml .= '<loc>' . htmlspecialchars($url) . '</loc>';
            $xml .= '<lastmod>' . date('Y-m-d') . '</lastmod>';
            $xml .= '<changefreq>weekly</changefreq>';
            $xml .= '<priority>0.8</priority>';
            $xml .= '</url>';
        }

        $xml .= '</urlset>';

        file_put_contents(public_path('sitemap.xml'), $xml);

        $this->info('Sitemap generated successfully at public/sitemap.xml');
    }
}
