<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Carbon\Carbon;

class AdminUserController extends Controller
{
    /**
     * Show approved users for a specific role with filters + pagination.
     */
    public function index(Request $request, string $role)
    {
        $allowedRoles = ['patient', 'doctor', 'volunteer', 'support', 'seo'];

        abort_unless(in_array($role, $allowedRoles), 404);

        // Beautiful label for heading
        $roleLabel = $this->getRoleLabel($role);

        // Base query: only approved users for this role
        $query = $this->buildFilteredQuery($request, $role);

        // For header stats
        $totalApproved = User::where('role', $role)
            ->where('is_approved', true)
            ->count();

        // Paginated result (20 per page)
        $users = $query->orderBy('created_at', 'desc')
            ->paginate(20)
            ->withQueryString();

        $filteredCount = $users->total();

        return view('admin.users.index', compact(
            'role',
            'roleLabel',
            'users',
            'totalApproved',
            'filteredCount'
        ));
    }

    /**
     * Export filtered users as CSV (respects same filters as index).
     */
    public function export(Request $request, string $role)
    {
        $allowedRoles = ['patient', 'doctor', 'volunteer', 'support', 'seo'];

        abort_unless(in_array($role, $allowedRoles), 404);

        $roleLabel = $this->getRoleLabel($role);

        $query = $this->buildFilteredQuery($request, $role);

        $filename = strtolower($role) . '_users_' . now()->format('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-Type'        => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $columns = ['ID', 'Name', 'Email', 'Phone', 'Role', 'Approved', 'Created At'];

        $callback = function () use ($query, $columns) {
            $handle = fopen('php://output', 'w');

            // Header row
            fputcsv($handle, $columns);

            // Stream rows in chunks (enterprise-friendly)
            $query->orderBy('created_at', 'desc')
                ->chunk(500, function ($users) use ($handle) {
                    foreach ($users as $user) {
                        fputcsv($handle, [
                            $user->id,
                            $user->name ?? '',
                            $user->email ?? '',
                            $user->phone ?? '',
                            $user->role,
                            $user->is_approved ? 'Yes' : 'No',
                            optional($user->created_at)->format('Y-m-d H:i'),
                        ]);
                    }
                });

            fclose($handle);
        };

        return response()->streamDownload($callback, $filename, $headers);
    }

    /**
     * Shared filtered query for index + export.
     *
     * Currently filters by registration date (created_at).
     * If later you add 'approved_at', you can switch to that.
     */
    protected function buildFilteredQuery(Request $request, string $role)
    {
        $query = User::where('role', $role)
            ->where('is_approved', true);

        // 1) TEXT SEARCH (name / email / phone)
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', '%' . $search . '%')
                    ->orWhere('email', 'like', '%' . $search . '%')
                    ->orWhere('phone', 'like', '%' . $search . '%');
            });
        }

        // 2) DATE RANGE FILTER
        $range = $request->input('range', 'all'); // all|today|this_week|this_month|custom

        $today = Carbon::today();

        if ($range === 'today') {
            $query->whereDate('created_at', $today);
        } elseif ($range === 'this_week') {
            $query->whereBetween('created_at', [
                $today->copy()->startOfWeek(),
                $today->copy()->endOfWeek(),
            ]);
        } elseif ($range === 'this_month') {
            $query->whereBetween('created_at', [
                $today->copy()->startOfMonth(),
                $today->copy()->endOfMonth(),
            ]);
        } elseif ($range === 'custom') {
            $start = $request->input('start_date');
            $end   = $request->input('end_date');

            if ($start && $end) {
                $startDate = Carbon::parse($start)->startOfDay();
                $endDate   = Carbon::parse($end)->endOfDay();

                $query->whereBetween('created_at', [$startDate, $endDate]);
            }
        }

        return $query;
    }

    /**
     * Pretty label for titles.
     */
    protected function getRoleLabel(string $role): string
    {
        return match ($role) {
            'patient'   => 'Patients',
            'doctor'    => 'Doctors',
            'volunteer' => 'Volunteers',
            'support'   => 'Support Team',
            'seo'       => 'SEO Team',
            default     => ucfirst($role),
        };
    }
}
