<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Event;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class EventAdminController extends Controller
{
    public function index(Request $request)
    {
        $filter = $request->get('filter', 'all');
        
        $query = Event::orderBy('display_order')->orderByDesc('start_datetime');
        
        if ($filter === 'upcoming') {
            $query->upcoming();
        } elseif ($filter === 'live') {
            $query->live();
        } elseif ($filter === 'done') {
            $query->done();
        }
        
        $events = $query->paginate(15);
        
        return view('admin.events.index', compact('events', 'filter'));
    }

    public function create()
    {
        return view('admin.events.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'slug' => ['nullable', 'string', 'max:255', 'unique:events,slug'],
            'description' => ['required', 'string'],
            'excerpt' => ['nullable', 'string'],
            'event_type' => ['required', 'string'],
            'event_mode' => ['required', 'string'],
            'start_datetime' => ['required', 'date'],
            'end_datetime' => ['required', 'date', 'after:start_datetime'],
            'timezone' => ['nullable', 'string'],
            'location' => ['nullable', 'string', 'max:255'],
            'meeting_link' => ['nullable', 'url', 'max:500'],
            'registration_link' => ['nullable', 'url', 'max:500'],
            'max_participants' => ['nullable', 'integer', 'min:1'],
            'image' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:2048'],
            'is_published' => ['nullable', 'boolean'],
            'is_featured' => ['nullable', 'boolean'],
            'display_order' => ['nullable', 'integer'],
        ]);

        // Auto-generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['title']);
            
            if (Event::where('slug', $data['slug'])->exists()) {
                $data['slug'] .= '-' . time();
            }
        }

        // Handle image upload
        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('events', 'public');
            $imagePath = 'storage/' . $imagePath;
        }

        $event = Event::create([
            'title' => $data['title'],
            'slug' => $data['slug'],
            'description' => $data['description'],
            'excerpt' => $data['excerpt'] ?? null,
            'event_type' => $data['event_type'],
            'event_mode' => $data['event_mode'],
            'start_datetime' => $data['start_datetime'],
            'end_datetime' => $data['end_datetime'],
            'timezone' => $data['timezone'] ?? 'Asia/Kolkata',
            'location' => $data['location'] ?? null,
            'meeting_link' => $data['meeting_link'] ?? null,
            'registration_link' => $data['registration_link'] ?? null,
            'max_participants' => $data['max_participants'] ?? null,
            'image_path' => $imagePath,
            'is_published' => $request->boolean('is_published', true),
            'is_featured' => $request->boolean('is_featured', false),
            'display_order' => $data['display_order'] ?? 0,
        ]);

        return redirect()
            ->route('admin.events.index')
            ->with('success', 'Event created successfully.');
    }

    public function edit(Event $event)
    {
        return view('admin.events.edit', compact('event'));
    }

    public function update(Request $request, Event $event)
    {
        $data = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'slug' => ['nullable', 'string', 'max:255', 'unique:events,slug,' . $event->id],
            'description' => ['required', 'string'],
            'excerpt' => ['nullable', 'string'],
            'event_type' => ['required', 'string'],
            'event_mode' => ['required', 'string'],
            'start_datetime' => ['required', 'date'],
            'end_datetime' => ['required', 'date', 'after:start_datetime'],
            'timezone' => ['nullable', 'string'],
            'location' => ['nullable', 'string', 'max:255'],
            'meeting_link' => ['nullable', 'url', 'max:500'],
            'registration_link' => ['nullable', 'url', 'max:500'],
            'max_participants' => ['nullable', 'integer', 'min:1'],
            'image' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:2048'],
            'is_published' => ['nullable', 'boolean'],
            'is_featured' => ['nullable', 'boolean'],
            'display_order' => ['nullable', 'integer'],
        ]);

        // Auto-generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['title']);
            
            if (Event::where('slug', $data['slug'])->where('id', '!=', $event->id)->exists()) {
                $data['slug'] .= '-' . time();
            }
        }

        // Handle image upload
        $imagePath = $event->image_path;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('events', 'public');
            $imagePath = 'storage/' . $imagePath;
        }

        $event->update([
            'title' => $data['title'],
            'slug' => $data['slug'],
            'description' => $data['description'],
            'excerpt' => $data['excerpt'] ?? null,
            'event_type' => $data['event_type'],
            'event_mode' => $data['event_mode'],
            'start_datetime' => $data['start_datetime'],
            'end_datetime' => $data['end_datetime'],
            'timezone' => $data['timezone'] ?? 'Asia/Kolkata',
            'location' => $data['location'] ?? null,
            'meeting_link' => $data['meeting_link'] ?? null,
            'registration_link' => $data['registration_link'] ?? null,
            'max_participants' => $data['max_participants'] ?? null,
            'image_path' => $imagePath,
            'is_published' => $request->boolean('is_published', true),
            'is_featured' => $request->boolean('is_featured', false),
            'display_order' => $data['display_order'] ?? 0,
        ]);

        return redirect()
            ->route('admin.events.index')
            ->with('success', 'Event updated successfully.');
    }

    public function destroy(Event $event)
    {
        $event->delete();

        return redirect()
            ->route('admin.events.index')
            ->with('success', 'Event deleted successfully.');
    }
}
