<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\LatestUpdate;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class LatestUpdateController extends Controller
{
    public function index()
    {
        $updates = LatestUpdate::orderByDesc('is_pinned')
            ->orderByDesc('published_at')
            ->orderByDesc('created_at')
            ->paginate(15);

        return view('admin.updates.index', compact('updates'));
    }

    public function create()
    {
        $categories = $this->categories();
        return view('admin.updates.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title'        => 'required|string|max:255',
            'category'     => 'nullable|string|max:80',
            'excerpt'      => 'nullable|string|max:300',
            'body'         => 'nullable|string',
            'is_published' => 'nullable|boolean',
            'is_pinned'    => 'nullable|boolean',
        ]);

        // Only treat as pinned when the checkbox was present in the request
        $data['is_published'] = $request->boolean('is_published');
        $data['is_pinned']    = $request->has('is_pinned') ? $request->boolean('is_pinned') : false;
        $data['slug']         = Str::slug($data['title']) . '-' . Str::random(5);

        if ($data['is_published'] && !isset($data['published_at'])) {
            $data['published_at'] = now();
        }

        LatestUpdate::create($data);

        return redirect()
            ->route('admin.updates.index')
            ->with('success', 'Update created successfully.');
    }

    public function edit(LatestUpdate $update)
    {
        $categories = $this->categories();
        return view('admin.updates.edit', compact('update', 'categories'));
    }

    public function update(Request $request, LatestUpdate $update)
    {
        $data = $request->validate([
            'title'        => 'required|string|max:255',
            'category'     => 'nullable|string|max:80',
            'excerpt'      => 'nullable|string|max:300',
            'body'         => 'nullable|string',
            'is_published' => 'nullable|boolean',
            'is_pinned'    => 'nullable|boolean',
        ]);

        // Only treat as pinned when the checkbox was present in the request
        $data['is_published'] = $request->boolean('is_published');
        $data['is_pinned']    = $request->has('is_pinned') ? $request->boolean('is_pinned') : false;

        // Only regen slug if title changed
        if ($update->title !== $data['title']) {
            $data['slug'] = Str::slug($data['title']) . '-' . Str::random(5);
        }

        if ($data['is_published'] && !$update->published_at) {
            $data['published_at'] = now();
        }

        $update->update($data);

        return redirect()
            ->route('admin.updates.index')
            ->with('success', 'Update updated successfully.');
    }

    public function destroy(LatestUpdate $update)
    {
        $update->delete();

        return redirect()
            ->route('admin.updates.index')
            ->with('success', 'Update deleted.');
    }

    protected function categories(): array
    {
        return [
            'Announcement',
            'Event',
            'Research',
            'Patient Story',
            'Clinical Update',
            'Platform Update',
        ];
    }
}
