<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MAB;
use App\Models\MabSession;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class MABController extends Controller
{
    public function index()
    {
        $members = MAB::with('session')->latest()->get();
        return view('admin.mab.index', compact('members'));
    }   

    public function create()
    {
        $sessions = MabSession::orderBy('created_at', 'desc')->get();
        return view('admin.mab.create', compact('sessions'));
    }   

    public function store(Request $request)
    {
        // \Log::info('MAB Store called', $request->all());

        $request->validate([
            'name' => 'required|string|max:255',
            'mab_session_id' => 'required|exists:mab_sessions,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'designation' => 'nullable|string|max:255',
            'location' => 'nullable|string|max:255',
            'about' => 'nullable|string',
        ]);

        try {
            $data = $request->except('image');

            if ($request->hasFile('image')) {
                $data['image'] = $request->file('image')->store('mab-members', 'public');
            }

            MAB::create($data);

            return redirect()->route('admin.mab.index')->with('success', 'Member added successfully.');
        } catch (\Exception $e) {
            // \Log::error('MAB Create Error: ' . $e->getMessage());
            return back()->withInput()->with('error', 'Error creating member: ' . $e->getMessage());
        }
    }

    public function edit(MAB $mab)
    {
        $sessions = MabSession::orderBy('created_at', 'desc')->get();
        return view('admin.mab.edit', compact('mab', 'sessions'));
    }   

    public function update(Request $request, MAB $mab)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'mab_session_id' => 'required|exists:mab_sessions,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'designation' => 'nullable|string|max:255',
            'location' => 'nullable|string|max:255',
            'about' => 'nullable|string',
        ]);

        $data = $request->except('image');

        if ($request->hasFile('image')) {
            if ($mab->image) {
                Storage::disk('public')->delete($mab->image);
            }
            $data['image'] = $request->file('image')->store('mab-members', 'public');
        }

        $mab->update($data);

        return redirect()->route('admin.mab.index')->with('success', 'Member updated successfully.');
    }

    public function destroy(MAB $mab)
    {
        if ($mab->image) {
            Storage::disk('public')->delete($mab->image);
        }
        $mab->delete();

        return redirect()->route('admin.mab.index')->with('success', 'Member deleted successfully.');
    }
}
