<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PatientStory;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PatientStoryAdminController extends Controller
{
    // List all stories (admin)
    public function index()
    {
        $stories = PatientStory::orderBy('display_order')->orderByDesc('created_at')->paginate(10);

        return view('admin.patient-stories.index', compact('stories'));
    }

    // Show create form
    public function create()
    {
        return view('admin.patient-stories.create');
    }

    // Store new story
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'       => ['required', 'string', 'max:255'],
            'condition'  => ['nullable', 'string', 'max:255'],
            'title'      => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'slug'       => ['nullable', 'string', 'max:255', 'unique:patient_stories,slug'],
            'image'      => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:2048'],
            'is_published' => ['nullable', 'boolean'],
            'display_order' => ['nullable', 'integer'],
        ]);

        // If slug not given, create from title
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['title']);

            // Make sure slug is unique
            if (PatientStory::where('slug', $data['slug'])->exists()) {
                $data['slug'] .= '-' . time();
            }
        }

        $imagePath = null;
        if ($request->hasFile('image')) {
            // store in storage/app/public/patient-stories
            $imagePath = $request->file('image')->store('patient-stories', 'public');
            // final path for asset()
            $imagePath = 'storage/' . $imagePath;
        } else {
            // fallback (optional)
            $imagePath = 'assets/images/pp1.jpg';
        }

        $story = PatientStory::create([
            'name'        => $data['name'],
            'condition'   => $data['condition'] ?? null,
            'title'       => $data['title'],
            'description' => $data['description'] ?? null,
            'slug'        => $data['slug'],
            'image_path'  => $imagePath,
            'is_published'=> $request->boolean('is_published', true),
            'display_order' => $data['display_order'] ?? 0,
        ]);

        return redirect()
            ->route('admin.patient-stories.index')
            ->with('success', 'Patient story created successfully.');
    }

    // Show edit form
    public function edit(PatientStory $patientStory)
    {
        return view('admin.patient-stories.edit', [
            'story' => $patientStory,
        ]);
    }

    // Update story
    public function update(Request $request, PatientStory $patientStory)
    {
        $data = $request->validate([
            'name'       => ['required', 'string', 'max:255'],
            'condition'  => ['nullable', 'string', 'max:255'],
            'title'      => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'slug'       => ['nullable', 'string', 'max:255', 'unique:patient_stories,slug,' . $patientStory->id],
            'image'      => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:2048'],
            'is_published' => ['nullable', 'boolean'],
            'display_order' => ['nullable', 'integer'],
        ]);

        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['title']);

            if (
                PatientStory::where('slug', $data['slug'])
                    ->where('id', '!=', $patientStory->id)
                    ->exists()
            ) {
                $data['slug'] .= '-' . time();
            }
        }

        $imagePath = $patientStory->image_path;

        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('patient-stories', 'public');
            $imagePath = 'storage/' . $imagePath;
        }

        $patientStory->update([
            'name'        => $data['name'],
            'condition'   => $data['condition'] ?? null,
            'title'       => $data['title'],
            'description' => $data['description'] ?? null,
            'slug'        => $data['slug'],
            'image_path'  => $imagePath,
            'is_published'=> $request->boolean('is_published', true),
            'display_order' => $data['display_order'] ?? 0,
        ]);

        return redirect()
            ->route('admin.patient-stories.index')
            ->with('success', 'Patient story updated successfully.');
    }

    // Delete story
    public function destroy(PatientStory $patientStory)
    {
        $patientStory->delete();

        return redirect()
            ->route('admin.patient-stories.index')
            ->with('success', 'Patient story deleted successfully.');
    }
}
