<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\YoutubeVideo;
use Illuminate\Http\Request;

class YouTubeVideoController extends Controller
{
    /**
     * Helper to get all language options.
     */
    protected function languageOptions(): array
    {
        return YoutubeVideo::languageOptions();
    }

    public function index(Request $request)
    {
        $languages = $this->languageOptions();

        // Optional filter by language for admin list
        $videosQuery = YoutubeVideo::orderBy('sort_order')
            ->orderByDesc('created_at');

        if ($request->filled('language')) {
            $videosQuery->where('language', $request->input('language'));
        }

        $videos = $videosQuery->paginate(20)
            ->appends($request->only('language'));

        return view('admin.youtube.index', compact('videos', 'languages'));
    }

    public function create()
    {
        $languages = $this->languageOptions();

        return view('admin.youtube.create', compact('languages'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'youtube_id'   => 'required|string|max:50',
            'title'        => 'required|string|max:255',
            'language'     => 'required|string|max:50',
            'meta'         => 'nullable|string|max:255',
            'is_featured'  => 'nullable|boolean',
            'is_published' => 'nullable|boolean',
            'sort_order'   => 'nullable|integer|min:0',
        ]);

        $data['is_featured']  = $request->boolean('is_featured');
        $data['is_published'] = $request->boolean('is_published');
        $data['sort_order']   = $data['sort_order'] ?? 0;

        // Ensure only one featured
        if ($data['is_featured']) {
            YoutubeVideo::where('is_featured', true)->update(['is_featured' => false]);
        }

        YoutubeVideo::create($data);

        return redirect()
            ->route('admin.youtube.index')
            ->with('success', 'YouTube video added successfully.');
    }

    public function edit(YoutubeVideo $youtubeVideo)
    {
        $languages = $this->languageOptions();

        return view('admin.youtube.edit', [
            'video'     => $youtubeVideo,
            'languages' => $languages,
        ]);
    }

    public function update(Request $request, YoutubeVideo $youtubeVideo)
    {
        $data = $request->validate([
            'youtube_id'   => 'required|string|max:50',
            'title'        => 'required|string|max:255',
            'language'     => 'required|string|max:50',
            'meta'         => 'nullable|string|max:255',
            'is_featured'  => 'nullable|boolean',
            'is_published' => 'nullable|boolean',
            'sort_order'   => 'nullable|integer|min:0',
        ]);

        $data['is_featured']  = $request->boolean('is_featured');
        $data['is_published'] = $request->boolean('is_published');
        $data['sort_order']   = $data['sort_order'] ?? 0;

        if ($data['is_featured']) {
            YoutubeVideo::where('is_featured', true)
                ->where('id', '!=', $youtubeVideo->id)
                ->update(['is_featured' => false]);
        }

        $youtubeVideo->update($data);

        return redirect()
            ->route('admin.youtube.index')
            ->with('success', 'YouTube video updated successfully.');
    }

    public function destroy(YoutubeVideo $youtubeVideo)
    {
        $youtubeVideo->delete();

        return redirect()
            ->route('admin.youtube.index')
            ->with('success', 'YouTube video deleted.');
    }
}
