<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LoginController extends Controller
{
    public function showLoginForm()
    {
        return view('auth.login');
    }

    protected function redirectToDashboard(User $user)
    {
        return match ($user->role) {
            User::ROLE_ADMIN      => route('admin.dashboard'),
            User::ROLE_SUPPORT    => route('support.dashboard'),
            User::ROLE_SEO        => route('seo.dashboard'),
            User::ROLE_DOCTOR     => route('doctor.dashboard'),
            User::ROLE_PATIENT    => route('patient.dashboard'),
            User::ROLE_VOLUNTEER  => route('volunteer.dashboard'),
            default               => route('home'),
        };
    }

 public function login(Request $request)
{
    $credentials = $request->validate([
        'email'    => ['required', 'email'],
        'password' => ['required'],
    ]);

    if (Auth::attempt($credentials)) {
        $request->session()->regenerate();

        // 🔒 Block users if not approved
        if (!auth()->user()->is_approved) {
            Auth::logout();
            return back()
                ->withErrors([
                    'email' => 'Your registration is pending admin approval. You will be able to log in once your account is approved.',
                ])
                ->onlyInput('email');
        }

        // Normal redirect based on role…
        $role = auth()->user()->role;

        if ($role === 'admin') {
            return redirect()->route('admin.dashboard');
        } elseif ($role === 'doctor') {
            return redirect()->route('doctor.dashboard');
        } elseif ($role === 'patient') {
            return redirect()->route('patient.dashboard');
        } elseif ($role === 'volunteer') {
            return redirect()->route('volunteer.dashboard');
        } elseif ($role === 'support') {
            return redirect()->route('support.dashboard');
        } elseif ($role === 'seo') {
            return redirect()->route('seo.dashboard');
        }
        

        return redirect()->route('home');
    }

    return back()->withErrors([
        'email' => 'The provided credentials do not match our records.',
    ])->onlyInput('email');
}

    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('home')
            ->with('success', 'You have been logged out.');
    }
}
