<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\PatientProfile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PatientRegisterController extends Controller
{
    public function showRegistrationForm()
    {
        return view('auth.patient-register');
    }

    public function register(Request $request)
    {
        // VALIDATION MATCHING YOUR MIGRATIONS
        $validated = $request->validate([
            'relationship_role' => ['required', 'in:patient,family_member'],
            'first_name'        => ['required', 'string', 'max:190'],
            'last_name'         => ['required', 'string', 'max:190'],

            'email'             => ['required', 'email', 'max:190', 'unique:users,email'],
            'phone'             => ['required', 'string', 'max:20'],

            'country'           => ['required', 'string', 'max:190'],
            'state'             => ['required', 'string', 'max:190'],
            'city'              => ['required', 'string', 'max:190'],

            'disease_id'        => ['required', 'integer', 'min:1'], // adjust later if you have diseases table

            'gender'            => ['required', 'in:male,female,other'],
            'age'               => ['required', 'integer', 'min:1', 'max:120'],

            'doctor_name'       => ['required', 'string', 'max:190'],
            'about'             => ['nullable', 'string', 'max:2000'],

            'password'          => ['required', 'confirmed', 'min:8'],
            'accept_terms'      => ['accepted'],

            // reCAPTCHA (anhskohbo/no-captcha)
            'g-recaptcha-response' => ['required', 'captcha'],
        ], [
            'g-recaptcha-response.required' => 'Please confirm you are not a robot.',
            'g-recaptcha-response.captcha'  => 'Captcha verification failed, please try again.',
            'accept_terms.accepted'         => 'You must accept the terms and conditions.',
        ]);

        // CREATE USER
        $user = User::create([
            'name'      => $validated['first_name'] . ' ' . $validated['last_name'],
            'email'     => $validated['email'],
            'phone'     => $validated['phone'],
            'role'      => 'patient',
            'is_active' => true,          // uska access verified hone ke baad hi full hoga
            'password'  => $validated['password'], // cast -> hashed
        ]);

        // CREATE PATIENT PROFILE
        PatientProfile::create([
            'user_id'          => $user->id,
            'relationship_role'=> $validated['relationship_role'],
            'first_name'       => $validated['first_name'],
            'last_name'        => $validated['last_name'],
            'phone'            => $validated['phone'],
            'country'          => $validated['country'],
            'state'            => $validated['state'],
            'city'             => $validated['city'],
            'disease_id'       => $validated['disease_id'],
            'gender'           => $validated['gender'],
            'age'              => $validated['age'],
            'doctor_name'      => $validated['doctor_name'],
            'about'            => $validated['about'] ?? '',
        ]);

        // LOGIN IMMEDIATELY (UNVERIFIED)
        Auth::login($user);

        // SEND EMAIL VERIFICATION
        $user->sendEmailVerificationNotification();

        // GO TO VERIFICATION PAGE
        return redirect()
            ->route('verification.notice')
            ->with('status', 'verification-link-sent');
    }
}
