<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\PatientProfile;
use App\Models\DoctorProfile;
use App\Models\VolunteerProfile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class RegistrationController extends Controller
{
    /* ---------- COMMON HELPERS ---------- */

    protected function generateCaptcha(Request $request): array
    {
        $a = rand(1, 9);
        $b = rand(1, 9);

        $request->session()->put('captcha_answer', $a + $b);

        return [
            'question' => "{$a} + {$b} = ?",
        ];
    }

    protected function validateCaptcha(Request $request): void
    {
        $expected = $request->session()->get('captcha_answer');
        $given    = $request->input('captcha_answer');

        if ($expected === null || (int) $given !== (int) $expected) {
            throw ValidationException::withMessages([
                'captcha_answer' => 'Incorrect answer to the security question. Please try again.',
            ]);
        }
    }

    /**
     * Send registration emails to user + admin.
     * Returns ['userMailSent' => bool]
     */
    protected function sendRegistrationEmails(User $user, string $roleLabel): array
    {
        $userMailSent = true;

        // 1) Mail to user
        try {
            // You can replace these views with your own email templates
            Mail::send('emails.registration-user', [
                'user'      => $user,
                'roleLabel' => $roleLabel,
            ], function ($message) use ($user) {
                $message->to($user->email, $user->name)
                    ->subject('Welcome to Myositis India');
            });
        } catch (\Throwable $e) {
            $userMailSent = false;
            Log::error('User registration mail failed: ' . $e->getMessage());
        }

        // 2) Mail to admin
        try {
            Mail::send('emails.registration-admin', [
                'user'      => $user,
                'roleLabel' => $roleLabel,
            ], function ($message) {
                $message->to('personal.sartajali@gmail.com', 'Myositis India Admin')
                    ->subject('New user registered on Myositis India');
            });
        } catch (\Throwable $e) {
            Log::error('Admin registration mail failed: ' . $e->getMessage());
        }

        return ['userMailSent' => $userMailSent];
    }

    /* ---------- ROLE SELECTION PAGE ---------- */

    public function showRoleSelection(Request $request)
    {
        return view('auth.register-role');
    }

    /* ---------- THANK YOU PAGE ---------- */

    public function showThankYou(Request $request)
    {
        return view('auth.thank-you');
    }

    /* ---------- PATIENT REGISTRATION ---------- */

    public function showPatientForm(Request $request)
    {
        $captcha = $this->generateCaptcha($request);

        return view('auth.register-patient', [
            'captchaQuestion' => $captcha['question'],
        ]);
    }

    public function registerPatient(Request $request)
    {
        // 1) Validate form input
        $validated = $request->validate(
            [
                'patient_type'   => 'required|in:patient,family_member',
                'fname'          => 'required|string|max:100',
                'lname'          => 'required|string|max:100',
                'phone'          => 'required|string|max:20',
                'email'          => 'required|email:rfc,dns|unique:users,email',
                'password'       => 'required|min:8|confirmed',
                'country'        => 'required|string|max:100',
                'state'          => 'required|string|max:100',
                'city'           => 'required|string|max:100',
                'disease_id'     => 'required|integer',
                'gender'         => 'required|in:male,female,other',
                'age'            => 'required|integer|min:0|max:120',
                'doctorName'     => 'required|string|max:190',
                'message'        => 'required|string|max:2000',
                'captcha_answer' => 'required|integer',
                'terms'          => 'accepted',
            ],
            [
                'patient_type.required' => 'Please select whether you are a patient or a family member.',
                'fname.required'        => 'Please enter your first name.',
                'lname.required'        => 'Please enter your last name.',
                'phone.required'        => 'Please enter a mobile number.',
                'email.required'        => 'Please enter your email address.',
                'email.email'           => 'Please enter a valid email address.',
                'email.unique'          => 'This email is already registered.',
                'password.required'     => 'Please choose a password.',
                'password.min'          => 'Your password should be at least :min characters.',
                'password.confirmed'    => 'Password and confirm password do not match.',
                'country.required'      => 'Please select your country.',
                'state.required'        => 'Please select your state.',
                'city.required'         => 'Please select your city.',
                'disease_id.required'   => 'Please select the type of myositis.',
                'gender.required'       => 'Please select your gender.',
                'age.required'          => 'Please enter your age.',
                'age.integer'           => 'Age should be a number.',
                'age.max'               => 'Please enter a realistic age (below :max).',
                'doctorName.required'   => 'Please enter your doctor’s name.',
                'message.required'      => 'Please tell us a little about yourself.',
                'captcha_answer.required' => 'Please solve the security question.',
                'captcha_answer.integer'  => 'Security answer must be a number.',
                'terms.accepted'        => 'You must accept the terms and conditions to register.',
            ]
        );

        // 2) Captcha check
        $this->validateCaptcha($request);

        // 3) Create user
        $user = User::create([
            'name'     => $validated['fname'] . ' ' . $validated['lname'],
            'email'    => $validated['email'],
            'phone'    => $validated['phone'],
            'password' => Hash::make($validated['password']),
            'role'     => User::ROLE_PATIENT,
            'is_active'=> true,
            'is_approved' => false, 
        ]);

        // 4) Create patient profile (aligns with your migration fields)
        PatientProfile::create([
            'user_id'          => $user->id,
            'relationship_role'=> $validated['patient_type'], // patient/family_member
            'first_name'       => $validated['fname'],
            'last_name'        => $validated['lname'],
            'phone'            => $validated['phone'],
            'country'          => $validated['country'],
            'state'            => $validated['state'],
            'city'             => $validated['city'],
            'disease_id'       => $validated['disease_id'],
            'gender'           => $validated['gender'],
            'age'              => $validated['age'],
            'doctor_name'      => $validated['doctorName'],
            'about'            => $validated['message'],
        ]);

        // 5) Send emails
        $mailResult = $this->sendRegistrationEmails($user, 'Patient');

        // 6) Auto login (optional, but you asked for login after register)
        Auth::login($user);

        // 7) Build human message
        $msg = 'You have been registered successfully as a patient.';
        $mailError = null;
        if (!$mailResult['userMailSent']) {
            $mailError = 'You are registered, but we could not send the confirmation email right now. Please check again later.';
        }

        // 8) Redirect to Thank You page, not dashboard
        return redirect()
            ->route('register.thankyou')
            ->with([
                'status'    => 'patient_registered',
                'success'   => $msg,
                'user_name' => $user->name,
                'mail_error'=> $mailError,
            ]);
    }

    /* ---------- DOCTOR / HEALTHCARE PROFESSIONAL ---------- */

    public function showDoctorForm(Request $request)
    {
        $captcha = $this->generateCaptcha($request);

        return view('auth.register-doctor', [
            'captchaQuestion' => $captcha['question'],
        ]);
    }

    public function registerDoctor(Request $request)
    {
        $validated = $request->validate(
            [
                'fname'            => 'required|string|max:100',
                'lname'            => 'required|string|max:100',
                'phone'            => 'required|string|max:20',
                'email'            => 'required|email:rfc,dns|unique:users,email',
                'password'         => 'required|min:8|confirmed',
                'country'          => 'required|string|max:100',
                'state'            => 'required|string|max:100',
                'city'             => 'required|string|max:100',
                'gender'           => 'required|in:male,female,other',
                'age'              => 'required|integer|min:20|max:100',
                'myositis_count'   => 'required|string|max:190', // "How many myositis patients..."
                'doctorSpeciality' => 'required|string|max:190',
                'message'          => 'required|string|max:2000',
                'captcha_answer'   => 'required|integer',
                'terms'            => 'accepted',
            ],
            [
                'fname.required'            => 'Please enter your first name.',
                'lname.required'            => 'Please enter your last name.',
                'phone.required'            => 'Please enter a mobile number.',
                'email.required'            => 'Please enter your email address.',
                'email.email'               => 'Please enter a valid email address.',
                'email.unique'              => 'This email is already registered.',
                'password.required'         => 'Please choose a password.',
                'password.min'              => 'Your password should be at least :min characters.',
                'password.confirmed'        => 'Password and confirm password do not match.',
                'country.required'          => 'Please select your country.',
                'state.required'            => 'Please select your state.',
                'city.required'             => 'Please select your city.',
                'gender.required'           => 'Please select your gender.',
                'age.required'              => 'Please enter your age.',
                'age.integer'               => 'Age should be a number.',
                'doctorSpeciality.required' => 'Please choose your speciality/area.',
                'myositis_count.required'   => 'Please indicate how many myositis patients you see in a year.',
                'message.required'          => 'Please tell us a little about yourself.',
                'captcha_answer.required'   => 'Please solve the security question.',
                'terms.accepted'            => 'You must accept the terms and conditions to register.',
            ]
        );

        $this->validateCaptcha($request);

        $user = User::create([
            'name'     => $validated['fname'] . ' ' . $validated['lname'],
            'email'    => $validated['email'],
            'phone'    => $validated['phone'],
            'password' => Hash::make($validated['password']),
            'role'     => User::ROLE_DOCTOR,
            'is_active'=> true,
            'is_approved' => false, 
        ]);

        // Align with your migration: specialization + clinic_name
        DoctorProfile::create([
            'user_id'        => $user->id,
            'first_name'     => $validated['fname'],
            'last_name'      => $validated['lname'],
            'phone'          => $validated['phone'],
            'country'        => $validated['country'],
            'state'          => $validated['state'],
            'city'           => $validated['city'],
            'specialization' => $validated['doctorSpeciality'],
            'gender'         => $validated['gender'],
            'age'            => $validated['age'],
            // store the "How many myositis patients..." answer here for now
            'clinic_name'    => $validated['myositis_count'],
            'about'          => $validated['message'],
        ]);

        $mailResult = $this->sendRegistrationEmails($user, 'Healthcare Professional');
        Auth::login($user);

        $msg = 'You have been registered successfully as a healthcare professional.';
        $mailError = null;
        if (!$mailResult['userMailSent']) {
            $mailError = 'You are registered, but we could not send the confirmation email right now. Please check again later.';
        }

        return redirect()
            ->route('register.thankyou')
            ->with([
                'status'    => 'doctor_registered',
                'success'   => $msg,
                'user_name' => $user->name,
                'mail_error'=> $mailError,
            ]);
    }

    /* ---------- VOLUNTEER REGISTRATION ---------- */

    public function showVolunteerForm(Request $request)
    {
        $captcha = $this->generateCaptcha($request);

        return view('auth.register-volunteer', [
            'captchaQuestion' => $captcha['question'],
        ]);
    }

    public function registerVolunteer(Request $request)
    {
        $validated = $request->validate(
            [
                'fname'          => 'required|string|max:100',
                'lname'          => 'required|string|max:100',
                'phone'          => 'required|string|max:20',
                'email'          => 'required|email:rfc,dns|unique:users,email',
                'password'       => 'required|min:8|confirmed',
                'country'        => 'required|string|max:100',
                'state'          => 'required|string|max:100',
                'city'           => 'required|string|max:100',
                'gender'         => 'required|in:male,female,other',
                'age'            => 'required|integer|min:12|max:100',
                'message'        => 'required|string|max:2000',
                'captcha_answer' => 'required|integer',
                'terms'          => 'accepted',
            ],
            [
                'fname.required'        => 'Please enter your first name.',
                'lname.required'        => 'Please enter your last name.',
                'phone.required'        => 'Please enter a mobile number.',
                'email.required'        => 'Please enter your email address.',
                'email.email'           => 'Please enter a valid email address.',
                'email.unique'          => 'This email is already registered.',
                'password.required'     => 'Please choose a password.',
                'password.min'          => 'Your password should be at least :min characters.',
                'password.confirmed'    => 'Password and confirm password do not match.',
                'country.required'      => 'Please select your country.',
                'state.required'        => 'Please select your state.',
                'city.required'         => 'Please select your city.',
                'gender.required'       => 'Please select your gender.',
                'age.required'          => 'Please enter your age.',
                'message.required'      => 'Please tell us a little about yourself.',
                'captcha_answer.required' => 'Please solve the security question.',
                'terms.accepted'        => 'You must accept the terms and conditions to register.',
            ]
        );

        $this->validateCaptcha($request);

        $user = User::create([
            'name'     => $validated['fname'] . ' ' . $validated['lname'],
            'email'    => $validated['email'],
            'phone'    => $validated['phone'],
            'password' => Hash::make($validated['password']),
            'role'     => User::ROLE_VOLUNTEER,
            'is_active'=> true,
            'is_approved' => false, 
        ]);

        VolunteerProfile::create([
            'user_id'    => $user->id,
            'first_name' => $validated['fname'],
            'last_name'  => $validated['lname'],
            'phone'      => $validated['phone'],
            'country'    => $validated['country'],
            'state'      => $validated['state'],
            'city'       => $validated['city'],
            'gender'     => $validated['gender'],
            'age'        => $validated['age'],
            'about'      => $validated['message'],
        ]);

        $mailResult = $this->sendRegistrationEmails($user, 'Volunteer');
        Auth::login($user);

        $msg = 'You have been registered successfully as a volunteer.';
        $mailError = null;
        if (!$mailResult['userMailSent']) {
            $mailError = 'You are registered, but we could not send the confirmation email right now. Please check again later.';
        }

        return redirect()
            ->route('register.thankyou')
            ->with([
                'status'    => 'volunteer_registered',
                'success'   => $msg,
                'user_name' => $user->name,
                'mail_error'=> $mailError,
            ]);
    }
}
