<?php

namespace App\Http\Controllers;

use App\Models\Event;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class EventController extends Controller
{
    public function index(Request $request)
    {
        $filter = $request->get('filter', 'all');
        $page = (int) $request->get('page', 1);
        
        $events = Cache::remember("events.{$filter}.page.{$page}", 300, function () use ($filter) {
            $query = Event::published()->orderBy('display_order')->orderBy('start_datetime');
            
            if ($filter === 'upcoming') {
                $query->upcoming();
            } elseif ($filter === 'live') {
                $query->live();
            } elseif ($filter === 'done') {
                $query->done();
            }
            
            return $query->paginate(12);
        });

        $seo = (object) [
            'title' => 'Events - Myositis India',
            'description' => 'Join our webinars, support groups, and events designed for the myositis community.',
            'keywords' => 'myositis events, webinars, support groups, india',
            'canonical_url' => url('/events'),
            'og_type' => 'website',
            'og_title' => 'Events - Myositis India',
            'og_description' => 'Upcoming events, webinars, and support groups for myositis patients and caregivers.',
            'og_image' => asset('assets/images/logo/myositisindialogo.webp'),
            'twitter_card' => 'summary_large_image',
            'twitter_title' => 'Events - Myositis India',
            'twitter_description' => 'Join our community events and support groups.',
            'twitter_image' => asset('assets/images/logo/myositisindialogo.webp'),
        ];

        return view('events.index', compact('events', 'filter', 'seo'));
    }

    public function show(Event $event)
    {
        $event = Cache::remember("event.{$event->slug}", 300, fn () => $event);

        $relatedEvents = Cache::remember("event.{$event->slug}.related", 300, function () use ($event) {
            return Event::published()
                ->where('id', '!=', $event->id)
                ->where('event_type', $event->event_type)
                ->orderBy('start_datetime')
                ->take(3)
                ->get();
        });

        $seo = (object) [
            'title' => $event->title . ' - Myositis India',
            'description' => $event->excerpt ?? Str::limit($event->description, 155),
            'keywords' => "myositis event, {$event->event_type}, india",
            'canonical_url' => url('/events/' . $event->slug),
            'og_type' => 'article',
            'og_title' => $event->title,
            'og_description' => $event->excerpt ?? Str::limit($event->description, 200),
            'og_image' => $event->image_path ? asset($event->image_path) : asset('assets/images/logo/myositisindialogo.webp'),
            'twitter_card' => 'summary_large_image',
            'twitter_title' => $event->title,
            'twitter_description' => $event->excerpt ?? Str::limit($event->description, 200),
            'twitter_image' => $event->image_path ? asset($event->image_path) : asset('assets/images/logo/myositisindialogo.webp'),
        ];

        return view('events.show', compact('event', 'relatedEvents', 'seo'));
    }
}
