<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\YoutubeVideo;
use App\Models\LatestUpdate;
use App\Models\PatientStory;
use App\Models\Event;
use Illuminate\Support\Facades\Cache;

class HomeController extends Controller
{
    /**
     * Homepage
     */
    public function index()
    {
        // Cache homepage aggregates to reduce TTFB
        $featured = Cache::remember('home.featured', 300, function () {
            $featured = YoutubeVideo::where('is_published', true)
                ->where('is_featured', true)
                ->orderBy('sort_order')
                ->first();

            if (! $featured) {
                $featured = YoutubeVideo::where('is_published', true)
                    ->orderBy('sort_order')
                    ->orderByDesc('created_at')
                    ->first();
            }

            return $featured;
        });

        $playlist = Cache::remember('home.playlist', 300, function () use ($featured) {
            $playlistQuery = YoutubeVideo::where('is_published', true)
                ->orderBy('sort_order')
                ->orderByDesc('created_at');

            if ($featured) {
                $playlistQuery->where('id', '!=', $featured->id);
            }

            return $playlistQuery->take(3)->get();
        });

        $headerUpdates = Cache::remember('home.headerUpdates', 300, function () {
            return LatestUpdate::published()
                ->where('is_pinned', true)
                ->orderByDesc('published_at')
                ->take(4)
                ->get();
        });

        $latestUpdates = Cache::remember('home.latestUpdates', 300, function () {
            return LatestUpdate::published()
                ->orderByDesc('published_at')
                ->orderByDesc('created_at')
                ->take(4)
                ->get();
        });

        $patientStories = Cache::remember('home.patientStories', 300, function () {
            return PatientStory::where('is_published', true)
                ->orderBy('display_order')
                ->orderByDesc('created_at')
                ->take(4)
                ->get();
        });

        $upcomingEvents = Cache::remember('home.upcomingEvents', 300, function () {
            return Event::published()
                ->upcoming()
                ->orderBy('display_order')
                ->orderBy('start_datetime')
                ->take(4)
                ->get();
        });

        $seo = (object) [
            'title' => 'Myositis India - Support, Care, Research & Awareness',
            'description' => 'Official website of Myositis India: patient helpline, expert network, clinical research updates, and caregiver resources.',
            'keywords' => 'myositis, autoimmune, patient support, helpline, research india',
            'robots' => 'index, follow',
            'canonical_url' => url('/'),
            'og_type' => 'website',
            'og_title' => 'Myositis India - Patient Care Foundation',
            'og_description' => 'Connect with specialists, access support groups, and explore the latest myositis research in India.',
            'og_image' => asset('assets/images/logo/myositisindialogo.webp'),
            'twitter_card' => 'summary_large_image',
            'twitter_title' => 'Myositis India - Patient Care Foundation',
            'twitter_description' => 'Support, awareness, and research for people living with myositis across India.',
            'twitter_image' => asset('assets/images/logo/myositisindialogo.webp'),
            'schema_js' => [
                '@context' => 'https://schema.org',
                '@type' => 'Organization',
                'name' => 'Myositis India',
                'url' => url('/'),
                'logo' => asset('assets/images/logo/myositisindialogo.webp'),
                'sameAs' => [
                    url('/'),
                ],
            ],
        ];

        // $mabSessions = Cache::remember('home.mabSessions', 300, function () {
            $mabSessions = \App\Models\MabSession::where('is_active', true)
                ->with(['members' => function($q) {
                     $q->latest();
                }])
                ->orderBy('title', 'desc')
                ->get();
        // });

        $gallery = \App\Models\Gallery::where('is_active', true)->orderBy('created_at', 'desc')->get();

        return view('welcome', compact(
            'featured',
            'playlist',
            'latestUpdates',
            'patientStories',
            'upcomingEvents',
            'headerUpdates',
            'seo',
            'mabSessions',
            'gallery'
        ));
    }

    /**
     * All videos page with language filter
     */
    public function videos(Request $request)
    {
        $cacheTtl = 300;

        // All languages that exist in DB
        $languages = Cache::remember('videos.languages', $cacheTtl, function () {
            return YoutubeVideo::where('is_published', true)
                ->select('language')
                ->distinct()
                ->orderBy('language')
                ->pluck('language')
                ->toArray();
        });

        $activeLanguage = $request->query('language');

        if ($activeLanguage && ! in_array($activeLanguage, $languages)) {
            // If invalid language passed, ignore it
            $activeLanguage = null;
        }

        // Decide which language tab is active
        if (!$activeLanguage && count($languages) > 0) {
            $activeLanguage = $languages[0]; // first language as default
        }

        $videos = Cache::remember("videos.list.{$activeLanguage}", $cacheTtl, function () use ($activeLanguage) {
            $videosQuery = YoutubeVideo::where('is_published', true)
                ->orderBy('sort_order')
                ->orderByDesc('created_at');

            if ($activeLanguage) {
                $videosQuery->where('language', $activeLanguage);
            }

            return $videosQuery->get();
        });

        // First video = main player
        $firstVideo = $videos->first();

        // Header updates (pinned only)
        $headerUpdates = Cache::remember('videos.headerUpdates', $cacheTtl, function () {
            return LatestUpdate::published()
                ->where('is_pinned', true)
                ->orderByDesc('published_at')
                ->take(4)
                ->get();
        });

        // Recent updates for sidebar/body
        $latestUpdates = Cache::remember('videos.latestUpdates', $cacheTtl, function () {
            return LatestUpdate::published()
                ->orderByDesc('published_at')
                ->orderByDesc('created_at')
                ->take(4)
                ->get();
        });

        $seo = (object) [
            'title' => 'Myositis India Videos - Patient Stories & Expert Talks',
            'description' => 'Watch patient stories, expert talks, educational content, and event recordings from Myositis India.',
            'keywords' => 'myositis videos, patient stories, expert talks, myositis education',
            'canonical_url' => url('/videos'),
            'og_type' => 'website',
            'og_title' => 'Myositis India Videos',
            'og_description' => 'Curated video library of patient stories and expert insights on myositis.',
            'og_image' => asset('assets/images/logo/myositisindialogo.webp'),
            'twitter_card' => 'summary_large_image',
            'twitter_title' => 'Myositis India Videos',
            'twitter_description' => 'Explore videos on myositis care, research, and patient experiences.',
            'twitter_image' => asset('assets/images/logo/myositisindialogo.webp'),
        ];

        return view('videos.index', [
            'languages'      => $languages,
            'activeLanguage' => $activeLanguage,
            'videos'         => $videos,
            'firstVideo'     => $firstVideo,
            'latestUpdates'  => $latestUpdates,
            'headerUpdates'  => $headerUpdates,
            'seo'            => $seo,
        ]);
    }

    /**
     * Public "Latest Updates" listing page
     */
    public function updates(Request $request)
    {
        $cacheTtl = 300;
        $page = (int) $request->get('page', 1);

        // Paginated full list for this page
        $updates = Cache::remember("updates.page.{$page}", $cacheTtl, function () {
            return LatestUpdate::published()
                ->orderByDesc('is_pinned')
                ->orderByDesc('published_at')
                ->orderByDesc('created_at')
                ->paginate(12);
        });

        // Header updates (pinned only)
        $headerUpdates = Cache::remember('updates.headerUpdates', $cacheTtl, function () {
            return LatestUpdate::published()
                ->where('is_pinned', true)
                ->orderByDesc('published_at')
                ->take(4)
                ->get();
        });

        // Recent updates for sidebar
        $latestUpdates = Cache::remember('updates.latestUpdates', $cacheTtl, function () {
            return LatestUpdate::published()
                ->orderByDesc('published_at')
                ->orderByDesc('created_at')
                ->take(4)
                ->get();
        });

        $seo = (object) [
            'title' => 'Myositis India Updates - News, Events, Research',
            'description' => 'Latest news, events, and research updates from Myositis India.',
            'keywords' => 'myositis news, research updates, events, india',
            'canonical_url' => url('/updates'),
            'og_type' => 'article',
            'og_title' => 'Myositis India Updates',
            'og_description' => 'Stay informed with news, events, and clinical updates on myositis.',
            'og_image' => asset('assets/images/logo/myositisindialogo.webp'),
            'twitter_card' => 'summary_large_image',
            'twitter_title' => 'Myositis India Updates',
            'twitter_description' => 'News and events from the Myositis India community.',
            'twitter_image' => asset('assets/images/logo/myositisindialogo.webp'),
        ];

        return view('updates.index', compact('updates', 'latestUpdates', 'headerUpdates', 'seo'));
    }

    /**
     * Show a single update (public)
     */
    public function showUpdate($slug)
    {
        $update = Cache::remember("update.slug.{$slug}", 300, function () use ($slug) {
            return \App\Models\LatestUpdate::published()
                ->where('slug', $slug)
                ->firstOrFail();
        });

        // Header updates (pinned only)
        $headerUpdates = Cache::remember('updates.headerUpdates', 300, function () {
            return \App\Models\LatestUpdate::published()
                ->where('is_pinned', true)
                ->orderByDesc('published_at')
                ->take(4)
                ->get();
        });

        // Recent updates for sidebar/other suggestions
        $latestUpdates = Cache::remember('updates.latestUpdates', 300, function () {
            return \App\Models\LatestUpdate::published()
                ->orderByDesc('published_at')
                ->orderByDesc('created_at')
                ->take(4)
                ->get();
        });

        $seo = (object) [
            'title' => $update->title ?? 'Myositis India Update',
            'description' => str($update->excerpt ?? $update->body ?? 'Myositis India update')->limit(155),
            'keywords' => 'myositis update, research, events, india',
            'canonical_url' => url('/updates/'.$update->slug),
            'og_type' => 'article',
            'og_title' => $update->title ?? 'Myositis India Update',
            'og_description' => str($update->excerpt ?? $update->body ?? 'Myositis India update')->limit(200),
            'og_image' => $update->cover_image ?? asset('assets/images/logo/myositisindialogo.webp'),
            'twitter_card' => 'summary_large_image',
            'twitter_title' => $update->title ?? 'Myositis India Update',
            'twitter_description' => str($update->excerpt ?? $update->body ?? 'Myositis India update')->limit(200),
            'twitter_image' => $update->cover_image ?? asset('assets/images/logo/myositisindialogo.webp'),
        ];

        return view('updates.show', compact('update', 'latestUpdates', 'headerUpdates', 'seo'));
    }


    public function samantha()
    {
        return view('brand-ambassador.samantha');
    }
}
