<?php

namespace App\Http\Controllers;

use App\Models\MabSession;
use App\Models\MAB;
use App\Models\SeoMetadatum;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class MabFrontendController extends Controller
{
    public function index()
    {
        // 1. Fetch metadata for this route if exists
        $seo = SeoMetadatum::where('route_name', 'mab.public.index')->first();

        // 2. Fetch all active sessions with members
        $sessions = MabSession::where('is_active', true)
            ->with(['members' => function($q) {
                $q->latest();
            }])
            ->orderBy('title', 'desc')
            ->get();

        return view('frontend.mab.index', compact('sessions', 'seo'));
    }

    public function show($id, $slug = null)
    {
        $mab = MAB::with('session')->findOrFail($id);

        // Security/SEO: redirect if slug doesn't match
        $expectedSlug = Str::slug($mab->name);
        if ($slug !== $expectedSlug) {
            return redirect()->route('mab.public.show', ['mab' => $mab->id, 'slug' => $expectedSlug], 301);
        }

        // Construct dynamic SEO object
        $seo = new \stdClass();
        $seo->title = $mab->name . ' - ' . ($mab->designation ?? 'MAB Member') . ' | Myositis India';
        $excerpt = Str::limit(strip_tags($mab->about), 150);
        $seo->description = $excerpt ?: 'Medical Advisory Board member profile for ' . $mab->name;
        $seo->keywords = 'myositis india, medical advisory board, ' . $mab->name . ', ' . ($mab->location ?? 'india');
        $seo->robots = 'index, follow';
        $seo->canonical_url = route('mab.public.show', ['mab' => $mab->id, 'slug' => $expectedSlug]);
        $seo->og_title = $seo->title;
        $seo->og_description = $seo->description;
        $seo->og_type = 'profile';
        if ($mab->image) {
            $seo->og_image = asset('storage/' . $mab->image);
            $seo->twitter_image = asset('storage/' . $mab->image);
        }
        $seo->twitter_card = 'summary_large_image';
        
        // No schema_js for now, or could add Person schema
        $seo->schema_js = null;

        return view('frontend.mab.show', compact('mab', 'seo'));
    }
}
