<?php

namespace App\Http\Controllers;

use App\Models\PatientStory;
use Illuminate\Support\Facades\Cache;

class PatientStoryController extends Controller
{
    // All stories page
    public function index()
    {
        $page = (int) request()->get('page', 1);
        $stories = Cache::remember("patientStories.page.{$page}", 300, function () {
            return PatientStory::where('is_published', true)
                ->orderBy('display_order')
                ->orderByDesc('created_at')
                ->paginate(12);
        });

        $seo = (object) [
            'title' => 'Patient Stories - Myositis India',
            'description' => 'Real experiences from myositis patients and caregivers across India.',
            'keywords' => 'myositis patient stories, experiences, india',
            'canonical_url' => url('/patient-stories'),
            'og_type' => 'website',
            'og_title' => 'Patient Stories - Myositis India',
            'og_description' => 'Read journeys of people living with myositis and their caregivers.',
            'og_image' => asset('assets/images/logo/myositisindialogo.webp'),
            'twitter_card' => 'summary_large_image',
            'twitter_title' => 'Patient Stories - Myositis India',
            'twitter_description' => 'Stories of resilience from the Myositis India community.',
            'twitter_image' => asset('assets/images/logo/myositisindialogo.webp'),
        ];

        return view('patient-stories.index', compact('stories', 'seo'));
    }

    // Single story page
    public function show(PatientStory $story)
    {
        $story = Cache::remember("patientStory.{$story->slug}", 300, fn () => $story);
        
        $imageUrl = $story->image_path ? asset($story->image_path) : asset('assets/images/logo/myositisindialogo.webp');
        $description = $story->description ?? ($story->excerpt ?? ($story->summary ?? ($story->body ?? 'Patient story from Myositis India community')));
        $descriptionText = is_string($description) ? $description : 'Patient story from Myositis India community';

        // Generate Article schema for Gen AI SEO
        $articleSchema = [
            '@context' => 'https://schema.org',
            '@type' => 'Article',
            'headline' => $story->title,
            'description' => mb_substr($descriptionText, 0, 200),
            'image' => [
                '@type' => 'ImageObject',
                'url' => $imageUrl,
                'width' => 1200,
                'height' => 630,
            ],
            'author' => [
                '@type' => 'Person',
                'name' => $story->name ?? 'Anonymous',
            ],
            'publisher' => [
                '@type' => 'Organization',
                'name' => 'Myositis India',
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => asset('assets/images/logo/myositisindialogo.webp'),
                ],
            ],
            'datePublished' => $story->created_at ? $story->created_at->toIso8601String() : now()->toIso8601String(),
            'dateModified' => $story->updated_at ? $story->updated_at->toIso8601String() : ($story->created_at ? $story->created_at->toIso8601String() : now()->toIso8601String()),
            'mainEntityOfPage' => [
                '@type' => 'WebPage',
                '@id' => url('/patient-stories/'.$story->slug),
            ],
            'keywords' => $story->condition ? ['myositis', $story->condition, 'patient story', 'india', 'healthcare'] : ['myositis', 'patient story', 'india', 'healthcare'],
        ];

        $seo = (object) [
            'title' => ($story->title ?? 'Patient Story') . ' - Myositis India',
            'description' => mb_substr($descriptionText, 0, 155),
            'keywords' => 'myositis patient story, ' . ($story->condition ?? '') . ', experience, india, ' . ($story->name ?? ''),
            'canonical_url' => url('/patient-stories/'.$story->slug),
            'og_type' => 'article',
            'og_title' => $story->title ?? 'Patient Story - Myositis India',
            'og_description' => mb_substr($descriptionText, 0, 200),
            'og_image' => $imageUrl,
            'twitter_card' => 'summary_large_image',
            'twitter_title' => $story->title ?? 'Patient Story - Myositis India',
            'twitter_description' => mb_substr($descriptionText, 0, 200),
            'twitter_image' => $imageUrl,
            'schema_js' => $articleSchema,
        ];

        return view('patient-stories.show', compact('story', 'seo'));
    }
}
