<?php

namespace App\Http\Controllers\Seo;

use Illuminate\Http\Request;
use App\Models\SeoMetadatum;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Controller;

class SeoDashboardController extends Controller
{
    public function index()
    {
        // Get all GET routes that have a name and are public-facing
        $routes = collect(Route::getRoutes())->filter(function ($route) {
            return in_array('GET', $route->methods())
                && $route->getName()
                && !str_starts_with($route->getName(), 'admin.')
                && !str_starts_with($route->getName(), 'ignition.');
        });

        // Fetch existing metadata keyed by route name
        $metadatas = SeoMetadatum::all()->keyBy('route_name');

        // Build page list with status scoring for dashboard
        $pages = $routes->map(function ($route) use ($metadatas) {
            $name = $route->getName();
            $meta = $metadatas->get($name);

            $hasTitle = !empty($meta?->title);
            $hasDesc = !empty($meta?->description);
            $hasOg = !empty($meta?->og_image);
            $hasCanonical = !empty($meta?->canonical_url);
            $isNoIndex = !empty($meta?->robots) && str_contains($meta->robots, 'noindex');

            $score = ($hasTitle + $hasDesc + $hasCanonical + $hasOg);

            if ($isNoIndex) {
                $status = 'noindex';
            } elseif ($score >= 4) {
                $status = 'perfect';
            } elseif ($score >= 2) {
                $status = 'partial';
            } else {
                $status = 'missing';
            }

            return [
                'name' => $name,
                'uri' => $route->uri(),
                'url' => url($route->uri()),
                'meta' => $meta,
                'status' => $status,
                'score' => $score,
                'hasTitle' => $hasTitle,
                'hasDesc' => $hasDesc,
                'hasOg' => $hasOg,
                'hasCanonical' => $hasCanonical,
                'isNoIndex' => $isNoIndex,
            ];
        })->values();

        $summary = [
            'total' => $pages->count(),
            'perfect' => $pages->where('status', 'perfect')->count(),
            'partial' => $pages->where('status', 'partial')->count(),
            'missing' => $pages->where('status', 'missing')->count(),
            'noindex' => $pages->where('status', 'noindex')->count(),
        ];

        return view('dashboards.seo', compact('pages', 'metadatas', 'summary'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'route_name'    => 'required|string',
            'title'         => 'nullable|string|max:255',
            'description'   => 'nullable|string',
            'keywords'      => 'nullable|string',
            'h1_heading'    => 'nullable|string|max:255',
            'canonical_url' => 'nullable|url',
            'robots'        => 'nullable|string',
            'priority'      => 'nullable|numeric|between:0,1',
            'change_frequency' => 'nullable|string',
            
            // Social
            'og_title'       => 'nullable|string',
            'og_description' => 'nullable|string',
            'og_image'       => 'nullable|string', // URL or path
            
            // Schema - we accept JSON string from frontend but Casts handle array.
            // However, the input is likely JSON string, so we might need to decode or let Eloquent handle it?
            // Since frontend sends JSON Object (from fetch body stringify), Laravel receives it as array/assoc if Content-Type is application/json.
            // Let's allow array or nullable.
            'schema_js'      => 'nullable', 
        ]);

        // Ensure schema_js is stored as array (for cast to work reliably)
        if (isset($validated['schema_js']) && is_string($validated['schema_js'])) {
             $decoded = json_decode($validated['schema_js'], true);
             // If valid JSON, use the array. If empty/invalid, null.
             $validated['schema_js'] = $decoded ?: null; 
        }

        SeoMetadatum::updateOrCreate(
            ['route_name' => $validated['route_name']],
            $validated
        );

        return response()->json(['message' => 'SEO updated successfully!']);
    }
}
