<?php

namespace App\Http\Controllers\Support;

use App\Http\Controllers\Controller;
use App\Models\Todo;
use App\Models\TodoCategory;
use App\Models\User;
use App\Notifications\TodoAssignedNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;

class TodoController extends Controller
{
    public function index(Request $request)
    {
        $filter = $request->get('filter', 'all');
        $category = $request->get('category');

        $query = Todo::with('category', 'volunteers', 'creator')->orderByDesc('created_at');

        if ($filter === 'open') {
            $query->whereIn('status', [Todo::STATUS_PENDING, Todo::STATUS_IN_PROGRESS, Todo::STATUS_BLOCKED]);
        } elseif ($filter === 'completed') {
            $query->where('status', Todo::STATUS_COMPLETED);
        }

        if ($category) {
            $query->where('category_id', $category);
        }

        // Category counts (total, open, completed)
        $categoryCounts = TodoCategory::withCount([
            'todos as total_tasks',
            'openTodos as open_tasks',
            'completedTodos as completed_tasks',
        ])->orderBy('display_order')->get();

        // Status filter (optional)
        $statusFilter = $request->get('status');
        if ($statusFilter) {
            $query->where('status', $statusFilter);
        }

        $todos = $query->paginate(20)->withQueryString();

        return view('support.todos.index', compact('todos', 'categoryCounts', 'filter', 'category'));
    }

    public function create()
    {
        $categories = TodoCategory::orderBy('display_order')->get();
        return view('support.todos.create', compact('categories'));
    }

    public function store(\App\Http\Requests\Support\StoreTodoRequest $request)
    {
        $data = $request->validated();

        $todo = Todo::create(array_merge($data, ['created_by' => auth()->id()]));

        // Assign volunteers if provided
        if (!empty($data['assignees'])) {
            foreach (array_unique($data['assignees']) as $volunteerId) {
                $todo->volunteers()->attach($volunteerId, ['assigned_by' => auth()->id(), 'assigned_at' => now()]);
            }

            // Notify assignees (in-app). For scale, use Notification::sendQueued in future.
            $volunteers = User::whereIn('id', $data['assignees'])->get();
            Notification::send($volunteers, new TodoAssignedNotification($todo));
        }

        return redirect()->route('support.todos.index')->with('success', 'Todo created successfully.');
    }

    public function show(Todo $todo)
    {
        $this->authorize('view', $todo);
        $todo->load('volunteers', 'updates.user', 'category', 'creator');
        $categories = TodoCategory::orderBy('display_order')->get();

        return view('support.todos.show', compact('todo', 'categories'));
    }

    public function edit(Todo $todo)
    {
        $this->authorize('update', $todo);
        $categories = TodoCategory::orderBy('display_order')->get();

        return view('support.todos.edit', compact('todo', 'categories'));
    }

    public function update(Request $request, Todo $todo)
    {
        $this->authorize('update', $todo);

        $data = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'category_id' => ['nullable', 'exists:todo_categories,id'],
            'priority' => ['required', 'in:low,medium,high,urgent'],
            'due_date' => ['nullable', 'date'],
            'status' => ['required', 'in:pending,in_progress,blocked,completed,cancelled'],
            'progress' => ['nullable', 'integer', 'min:0', 'max:100'],
        ]);

        $todo->update($data);

        return redirect()->route('support.todos.show', $todo)->with('success', 'Todo updated.');
    }

    public function destroy(Todo $todo)
    {
        $this->authorize('delete', $todo);
        $todo->delete();

        return redirect()->route('support.todos.index')->with('success', 'Todo deleted.');
    }

    /**
     * Assign/unassign volunteers (expects assignees array)
     */
    public function assign(\App\Http\Requests\Support\AssignTodoRequest $request, Todo $todo)
    {
        $this->authorize('assign', $todo);

        $data = $request->validated();

        $assignees = array_unique($data['assignees'] ?? []);

        // Sync while retaining assigned_by and assigned_at (attach new, detach removed)
        $current = $todo->volunteers()->pluck('users.id')->toArray();

        $toAttach = array_diff($assignees, $current);
        $toDetach = array_diff($current, $assignees);

        foreach ($toAttach as $userId) {
            $todo->volunteers()->attach($userId, ['assigned_by' => auth()->id(), 'assigned_at' => now()]);
        }

        if (!empty($toDetach)) {
            $todo->volunteers()->detach($toDetach);
        }

        if (!empty($toAttach)) {
            $volunteers = User::whereIn('id', $toAttach)->get();
            Notification::send($volunteers, new TodoAssignedNotification($todo));
        }

        return redirect()->back()->with('success', 'Assignments updated.');
    }

    /**
     * Simple server-side volunteer search for typeahead
     */
    public function searchVolunteers(Request $request)
    {
        $q = $request->get('q');

        $query = User::where('role', 'volunteer')->where('is_approved', true)->where('is_active', true);

        if ($q) {
            $query->where(function ($s) use ($q) {
                $s->where('name', 'like', "%{$q}%")
                  ->orWhere('email', 'like', "%{$q}%")
                  ->orWhere('phone', 'like', "%{$q}%");
            });
        }

        $results = $query->select('id', 'name', 'email', 'phone')->limit(20)->get();

        return response()->json($results);
    }
}
