<?php

namespace App\Http\Controllers\Volunteer;

use App\Http\Controllers\Controller;
use App\Models\Todo;
use App\Models\TodoUpdate;
use Illuminate\Http\Request;

class TodoController extends Controller
{
    public function index(Request $request)
    {
        $query = Todo::with('category', 'creator')
            ->assignedTo(auth()->id())
            ->orderBy('due_date', 'asc');

        $todos = $query->paginate(15)->withQueryString();

        return view('volunteer.todos.index', compact('todos'));
    }

    public function show(Todo $todo)
    {
        $this->authorize('view', $todo);
        $todo->load('volunteers', 'updates.user', 'category', 'creator');

        return view('volunteer.todos.show', compact('todo'));
    }

    public function updateStatus(Request $request, Todo $todo)
    {
        $this->authorize('update', $todo);

        $data = $request->validate([
            'status' => ['required', 'in:pending,in_progress,blocked,completed,cancelled'],
            'progress' => ['nullable', 'integer', 'min:0', 'max:100'],
        ]);

        $todo->update([
            'status' => $data['status'],
            'progress' => $data['progress'] ?? $todo->progress,
        ]);

        // store an update
        TodoUpdate::create([
            'todo_id' => $todo->id,
            'user_id' => auth()->id(),
            'type' => 'status',
            'content' => "Status set to {$data['status']}",
            'progress' => $data['progress'] ?? null,
        ]);

        return redirect()->back()->with('success', 'Status updated.');
    }

    /**
     * Mark a todo as done (completed) by volunteer
     */
    public function markDone(Request $request, Todo $todo)
    {
        $this->authorize('update', $todo);

        if ($todo->status === Todo::STATUS_COMPLETED) {
            return redirect()->back()->with('info', 'Task already completed.');
        }

        $todo->update([
            'status' => Todo::STATUS_COMPLETED,
            'progress' => 100,
        ]);

        TodoUpdate::create([
            'todo_id' => $todo->id,
            'user_id' => auth()->id(),
            'type' => 'status',
            'content' => 'Marked as completed',
            'progress' => 100,
        ]);

        // Notify creator (support)
        if ($todo->creator) {
            $todo->creator->notify(new \App\Notifications\TodoCompletedNotification($todo, auth()->user()));
        }

        return redirect()->back()->with('success', 'Marked as completed.');
    }

    public function storeUpdate(Request $request, Todo $todo)
    {
        $this->authorize('update', $todo);

        $data = $request->validate([
            'content' => ['required', 'string'],
            'progress' => ['nullable', 'integer', 'min:0', 'max:100'],
        ]);

        TodoUpdate::create([
            'todo_id' => $todo->id,
            'user_id' => auth()->id(),
            'type' => 'comment',
            'content' => $data['content'],
            'progress' => $data['progress'] ?? null,
        ]);

        return redirect()->back()->with('success', 'Update posted.');
    }
}
