<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Event extends Model
{
    protected $fillable = [
        'title',
        'slug',
        'description',
        'excerpt',
        'event_type',
        'event_mode',
        'start_datetime',
        'end_datetime',
        'timezone',
        'location',
        'meeting_link',
        'registration_link',
        'max_participants',
        'image_path',
        'is_published',
        'is_featured',
        'display_order',
    ];

    protected $casts = [
        'start_datetime' => 'datetime',
        'end_datetime' => 'datetime',
        'is_published' => 'boolean',
        'is_featured' => 'boolean',
    ];

    // Route key
    public function getRouteKeyName(): string
    {
        return 'slug';
    }

    // Scopes
    public function scopePublished($query)
    {
        return $query->where('is_published', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopeUpcoming($query)
    {
        return $query->where('end_datetime', '>', now());
    }

    public function scopeLive($query)
    {
        return $query->where('start_datetime', '<=', now())
                     ->where('end_datetime', '>=', now());
    }

    public function scopeDone($query)
    {
        return $query->where('end_datetime', '<', now());
    }

    // Accessors
    public function getStatusAttribute()
    {
        $now = now();
        
        if ($now->lt($this->start_datetime)) {
            return 'upcoming';
        } elseif ($now->gte($this->start_datetime) && $now->lte($this->end_datetime)) {
            return 'live';
        } else {
            return 'done';
        }
    }

    public function getStatusBadgeAttribute()
    {
        $status = $this->status;
        
        $badges = [
            'upcoming' => '<span class="inline-flex items-center px-2 py-0.5 rounded-full text-xs font-semibold bg-blue-100 text-blue-700">Upcoming</span>',
            'live' => '<span class="inline-flex items-center px-2 py-0.5 rounded-full text-xs font-semibold bg-red-100 text-red-700 animate-pulse">Live Now</span>',
            'done' => '<span class="inline-flex items-center px-2 py-0.5 rounded-full text-xs font-semibold bg-gray-100 text-gray-600">Completed</span>',
        ];
        
        return $badges[$status] ?? '';
    }

    public function getButtonTextAttribute()
    {
        return match($this->status) {
            'upcoming' => 'Register Now',
            'live' => 'Join Live',
            'done' => 'View Recording',
        };
    }

    public function getButtonStyleAttribute()
    {
        return match($this->status) {
            'upcoming' => 'background: var(--brand-blue);',
            'live' => 'background: #dc2626;',
            'done' => 'background: #6b7280;',
        };
    }

    public function getFormattedDateAttribute()
    {
        return $this->start_datetime->format('F d, Y');
    }

    public function getFormattedTimeAttribute()
    {
        return $this->start_datetime->format('g:i A') . ' – ' . $this->end_datetime->format('g:i A T');
    }

    public function getDayNameAttribute()
    {
        return $this->start_datetime->format('D');
    }

    public function getDayNumberAttribute()
    {
        return $this->start_datetime->format('d');
    }

    public function getMonthNameAttribute()
    {
        return $this->start_datetime->format('M');
    }
}
