<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class LatestUpdate extends Model
{
    protected $fillable = [
        'title',
        'slug',
        'category',
        'excerpt',
        'body',
        'is_published',
        'is_pinned',
        'published_at',
    ];

    protected $casts = [
        'is_published' => 'boolean',
        'is_pinned'    => 'boolean',
        'published_at' => 'datetime',
    ];

    // Auto-slug if not given & Cache clearing
    protected static function booted()
    {
        static::creating(function ($update) {
            if (empty($update->slug)) {
                $update->slug = Str::slug($update->title) . '-' . Str::random(5);
            }
            if (empty($update->published_at) && $update->is_published) {
                $update->published_at = now();
            }
        });

        static::updating(function ($update) {
            if ($update->is_published && empty($update->published_at)) {
                $update->published_at = now();
            }
        });

        // Clear cache on any modification
        static::saved(function () {
            \Illuminate\Support\Facades\Cache::forget('home.headerUpdates');
            \Illuminate\Support\Facades\Cache::forget('home.latestUpdates');
            \Illuminate\Support\Facades\Cache::forget('updates.headerUpdates');
            \Illuminate\Support\Facades\Cache::forget('updates.latestUpdates');
        });

        static::deleted(function () {
            \Illuminate\Support\Facades\Cache::forget('home.headerUpdates');
            \Illuminate\Support\Facades\Cache::forget('home.latestUpdates');
            \Illuminate\Support\Facades\Cache::forget('updates.headerUpdates');
            \Illuminate\Support\Facades\Cache::forget('updates.latestUpdates');
        });
    }

    // Scope: only published
    public function scopePublished($query)
    {
        return $query->where('is_published', true)
                     ->whereNotNull('published_at')
                     ->where('published_at', '<=', now());
    }
}
