<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Todo extends Model
{
    use HasFactory;

    public const PRIORITY_LOW = 'low';
    public const PRIORITY_MEDIUM = 'medium';
    public const PRIORITY_HIGH = 'high';
    public const PRIORITY_URGENT = 'urgent';

    public const STATUS_PENDING = 'pending';
    public const STATUS_IN_PROGRESS = 'in_progress';
    public const STATUS_BLOCKED = 'blocked';
    public const STATUS_COMPLETED = 'completed';
    public const STATUS_CANCELLED = 'cancelled';

    protected $fillable = [
        'title',
        'description',
        'category_id',
        'priority',
        'status',
        'progress',
        'due_date',
        'created_by',
    ];

    protected $casts = [
        'due_date' => 'datetime',
    ];

    // relations
    public function category()
    {
        // Explicit foreign key name matches migration column 'category_id'
        return $this->belongsTo(TodoCategory::class, 'category_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function volunteers()
    {
        return $this->belongsToMany(User::class, 'todo_volunteer', 'todo_id', 'volunteer_id')
            ->withPivot(['assigned_by', 'assigned_at'])
            ->withTimestamps();
    }

    public function updates()
    {
        return $this->hasMany(TodoUpdate::class);
    }

    // Scope: todos assigned to a volunteer
    public function scopeAssignedTo($query, $userId)
    {
        return $query->whereHas('volunteers', function ($q) use ($userId) {
            $q->where('users.id', $userId);
        });
    }
}
