<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable implements MustVerifyEmail
{
    use Notifiable;

    
    // Role constants for user types (match migration enum values)
    public const ROLE_ADMIN     = 'admin';
    public const ROLE_SUPPORT   = 'support';
    public const ROLE_SEO       = 'seo';
    public const ROLE_DOCTOR    = 'doctor';
    public const ROLE_PATIENT   = 'patient';
    public const ROLE_VOLUNTEER = 'volunteer';

    protected $fillable = [
        'name',
        'email',
        'role',
        'is_active',
        'phone',
        'password',
        'is_approved',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'is_active'         => 'boolean',
        'password'          => 'hashed', 
        'is_approved'       => 'boolean',
    ];

    // relationships
    public function patientProfile()
    {
        return $this->hasOne(PatientProfile::class);
    }

    public function doctorProfile()
    {
        return $this->hasOne(DoctorProfile::class);
    }

    public function volunteerProfile()
    {
        return $this->hasOne(VolunteerProfile::class);
    }

    /**
     * Todos assigned to this user (volunteer)
     */
    public function assignedTodos()
    {
        return $this->belongsToMany(Todo::class, 'todo_volunteer', 'volunteer_id', 'todo_id')
            ->withPivot(['assigned_by', 'assigned_at'])
            ->withTimestamps();
    }

    /**
     * Todos created by this user (support)
     */
    public function todosCreated()
    {
        return $this->hasMany(Todo::class, 'created_by');
    }

    /**
     * Todo updates (comments/status changes) by this user
     */
    public function todoUpdates()
    {
        return $this->hasMany(TodoUpdate::class);
    }
}
