<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Compliances\Adt1Compliance;
use App\Models\Core\Company;
use App\Models\Core\FinancialSession;
use Illuminate\Http\Request;
use App\Services\Adt1ComplianceService;
use App\Http\Requests\Admin\ImportAdt1CsvRequest;
use App\Http\Requests\Admin\StoreAdt1ComplianceRequest;

class Adt1ComplianceController extends Controller
{
    protected $adt1Service;

    public function __construct(Adt1ComplianceService $adt1Service)
    {
        $this->adt1Service = $adt1Service;
    }

    public function index(Request $request)
    {
        $this->authorize('viewAny', Adt1Compliance::class);

        // 1. Get Selected Session (or default to Active)
        $selectedSessionId = $request->get('session_id');
        if (!$selectedSessionId) {
            $activeSession = FinancialSession::where('is_active', true)->first();
            $selectedSessionId = $activeSession ? $activeSession->id : FinancialSession::latest()->first()->id;
        }

        // 2. Query Companies linked to this Session
        // We need ALL companies so we can show "Pending" for those without a record
        $query = Company::whereHas('financialSessions', function ($q) use ($selectedSessionId) {
            $q->where('financial_sessions.id', $selectedSessionId);
        })->with([
                    'adt1Compliances' => function ($q) use ($selectedSessionId) {
                        $q->where('financial_session_id', $selectedSessionId);
                    }
                ]);

        // Search Filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('cin', 'like', "%{$search}%");
            });
        }

        // Status Filter
        if ($request->filled('status')) {
            $status = $request->status;
            if ($status === 'Filed') {
                $query->whereHas('adt1Compliances', function ($q) use ($selectedSessionId) {
                    $q->where('financial_session_id', $selectedSessionId)
                        ->where('status', 'Filed');
                });
            } elseif ($status === 'Pending') {
                $query->where(function ($q) use ($selectedSessionId) {
                    $q->whereDoesntHave('adt1Compliances', function ($sub) use ($selectedSessionId) {
                        $sub->where('financial_session_id', $selectedSessionId);
                    })->orWhereHas('adt1Compliances', function ($sub) use ($selectedSessionId) {
                        $sub->where('financial_session_id', $selectedSessionId)
                            ->where('status', 'Pending');
                    });
                });
            } elseif ($status === 'Overdue') {
                $query->whereHas('adt1Compliances', function ($q) use ($selectedSessionId) {
                    $q->where('financial_session_id', $selectedSessionId)
                        ->where('status', 'Overdue');
                });
            }
        }

        $companies = $query->orderBy('name')->paginate(20);

        // Stats Calculation
        $statsSessionId = $selectedSessionId;
        $totalCompanies = Company::whereHas('financialSessions', function ($q) use ($statsSessionId) {
            $q->where('financial_sessions.id', $statsSessionId);
        })->count();

        // Count companies that have a Filed record for this session
        $filedCount = Company::whereHas('financialSessions', function ($q) use ($statsSessionId) {
            $q->where('financial_sessions.id', $statsSessionId);
        })->whereHas('adt1Compliances', function ($q) use ($statsSessionId) {
            $q->where('financial_session_id', $statsSessionId)
                ->where('status', 'Filed');
        })->count();

        $pendingCount = $totalCompanies - $filedCount;

        // Data for filters
        $sessions = FinancialSession::orderBy('name', 'desc')->get();
        $currentSession = FinancialSession::find($selectedSessionId);

        if ($request->ajax()) {
            return view('admin.adt1.partials.table', compact('companies', 'currentSession'))->render();
        }

        return view('admin.adt1.index', compact('companies', 'sessions', 'currentSession', 'filedCount', 'pendingCount', 'totalCompanies'));
    }

    public function edit(Company $company, FinancialSession $session)
    {
        $this->authorize('create', Adt1Compliance::class);

        // Find existing record or return null (for create mode)
        $record = Adt1Compliance::where('company_id', $company->id)
            ->where('financial_session_id', $session->id)
            ->first();

        // Pass compliance type ID for ADT-1
        $complianceType = \App\Models\Core\ComplianceType::where('code', 'ADT1')->first();

        return view('admin.adt1.edit', compact('company', 'session', 'record', 'complianceType'));
    }

    public function store(StoreAdt1ComplianceRequest $request, Company $company, FinancialSession $session)
    {
        // Use Service to update or create the record
        $this->adt1Service->updateCompliance($company, $session, $request->validated());

        return redirect()->route('admin.adt1.index', ['session_id' => $session->id])
            ->with('success', 'Compliance record updated successfully for ' . $company->name);
    }

    public function downloadSampleCsv()
    {
        $this->authorize('viewAny', Adt1Compliance::class);

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="adt1_import_sample.csv"',
            'Pragma' => 'no-cache',
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Expires' => '0',
        ];

        // Headers match Form Input Names exactly
        $columns = [
            'CIN', // 0
            'status', // 1
            'date_of_filing', // 2
            'srn', // 3
            'date_of_appointment', // 4
            'agm_held', // 5
            'from_year', // 6
            'to_year', // 7
            'no_of_years', // 8
            // 'due_date', <--- REMOVED (System Calculated)
            'name_of_auditor_firm', // 9 (Shifted)
            'name_of_signing_auditor', // 10 (Shifted)
            'checked_by' // 11 (New)
        ];

        $callback = function () use ($columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            $statuses = ['Filed', 'Pending'];

            // Generate 2 rows. Force first one to be 'Filed' to show full data.
            for ($i = 1; $i <= 2; $i++) {
                $status = ($i === 1) ? 'Filed' : 'Pending';

                // Use d/m/Y format for CSV Sample
                $dateFil = date('d/m/Y', strtotime("-{$i} days"));

                // Logic: Appointment usually before Filing
                $dateApp = date('d/m/Y', strtotime("-{$i} weeks - 1 month"));
                $dateAgm = date('d/m/Y', strtotime("-{$i} months"));

                // Tenure Dates
                $tsApp = strtotime("-{$i} weeks - 1 month");
                $tsTo = strtotime('+5 years', $tsApp);

                $fromDate = $dateApp;
                $toDate = date('d/m/Y', $tsTo);

                // Match exact CIN generation from CompanyController
                $cin = 'U' . (10000 + $i) . 'DL202' . ($i % 5) . 'PTC' . (100000 + $i);

                $row = [
                    $cin,                       // 0: CIN
                    $status,                    // 1: status
                    ($status === 'Filed') ? $dateFil : '', // 2: date_of_filing
                    ($status === 'Filed') ? 'R' . (90000000 + $i) : '', // 3: srn
                    $dateApp,                   // 4: date_of_appointment
                    $dateAgm,                   // 5: agm_held
                    $fromDate,                  // 6: from_year
                    $toDate,                    // 7: to_year
                    '5',                        // 8: no_of_years
                    // NO DUE DATE COLUMN
                    'Firm ' . chr(65 + ($i % 26)) . ' & Associates', // 9: name_of_auditor_firm
                    'CA. Auditor ' . $i,        // 10: name_of_signing_auditor
                    'Employee ' . $i            // 11: checked_by
                ];

                // Sanitize Formula Injection Risks
                $sanitizedRow = array_map(function ($field) {
                    if (is_string($field) && preg_match('/^[\=\+\-\@]/', $field)) {
                        return "'" . $field;
                    }
                    return $field;
                }, $row);

                fputcsv($file, $sanitizedRow);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function importCsv(ImportAdt1CsvRequest $request)
    {
        $file = $request->file('csv_file');
        $path = $file->getRealPath();
        $sessionId = $request->financial_session_id;

        try {
            $result = $this->adt1Service->importCompliances($path, $sessionId);

            // $result contains imported and failed counts
            $message = "Imported {$result['imported']} records successfully.";
            if ($result['failed'] > 0) {
                $message .= " Skipped {$result['failed']} records (Company not found).";
            }

            return redirect()->route('admin.adt1.index', ['session_id' => $sessionId])
                ->with('success', $message);

        } catch (\Exception $e) {
            return redirect()->route('admin.adt1.index', ['session_id' => $sessionId])
                ->with('error', 'Import failed: ' . $e->getMessage());
        }
    }
    public function export(Request $request)
    {
        $this->authorize('viewAny', Adt1Compliance::class);

        // 1. Get Selected Session (or default to Active)
        $selectedSessionId = $request->get('session_id');
        if (!$selectedSessionId) {
            $activeSession = FinancialSession::where('is_active', true)->first();
            $selectedSessionId = $activeSession ? $activeSession->id : FinancialSession::latest()->first()->id;
        }

        // 2. Query Companies linked to this Session
        // MATCHING INDEX LOGIC EXACTLY
        $query = Company::whereHas('financialSessions', function ($q) use ($selectedSessionId) {
            $q->where('financial_sessions.id', $selectedSessionId);
        })->with([
                    'adt1Compliances' => function ($q) use ($selectedSessionId) {
                        $q->where('financial_session_id', $selectedSessionId);
                    },
                    'financialSessions' => function ($q) use ($selectedSessionId) {
                        $q->where('financial_sessions.id', $selectedSessionId);
                    }
                ]);

        // Search Filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('cin', 'like', "%{$search}%");
            });
        }

        // Status Filter
        if ($request->filled('status')) {
            $status = $request->status;
            if ($status === 'Filed') {
                $query->whereHas('adt1Compliances', function ($q) use ($selectedSessionId) {
                    $q->where('financial_session_id', $selectedSessionId)
                        ->where('status', 'Filed');
                });
            } elseif ($status === 'Pending') {
                $query->where(function ($q) use ($selectedSessionId) {
                    $q->whereDoesntHave('adt1Compliances', function ($sub) use ($selectedSessionId) {
                        $sub->where('financial_session_id', $selectedSessionId);
                    })->orWhereHas('adt1Compliances', function ($sub) use ($selectedSessionId) {
                        $sub->where('financial_session_id', $selectedSessionId)
                            ->where('status', 'Pending');
                    });
                });
            } elseif ($status === 'Overdue') {
                $query->whereHas('adt1Compliances', function ($q) use ($selectedSessionId) {
                    $q->where('financial_session_id', $selectedSessionId)
                        ->where('status', 'Overdue');
                });
            }
        }

        // Get matching companies
        $companies = $query->orderBy('name')->get();
        $session = FinancialSession::find($selectedSessionId);

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="adt1_export_' . $session->name . '_' . date('Y-m-d') . '.csv"',
            'Pragma' => 'no-cache',
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Expires' => '0',
        ];

        $columns = [
            'Company Name',
            'CIN',
            'Financial Year',
            'Filing Status',
            'Date of Filing',
            'SRN',
            'Date of Appointment',
            'Date of AGM',
            'From Date',
            'To Date',
            'Tenure (Years)',
            'Due Filing Date',
            'Auditor Firm',
            'Signing Partner',
            'Checked By'
            // 'Created At',
            // 'Updated At'
        ];

        $callback = function () use ($companies, $columns, $session) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($companies as $company) {
                $record = $company->adt1Compliances->first(); // Since we eager loaded with where session_id

                // Determine Status
                $status = 'Pending';
                if ($record) {
                    $status = $record->status;
                }

                // Format Dates Helper
                $fmtDate = fn($d) => $d ? $d->format('d/m/Y') : '';

                // Extract Data
                $row = [
                    $company->name,
                    $company->cin,
                    $session->name,
                    $status,
                    $record ? $fmtDate($record->date_of_filing) : '',
                    $record ? $record->srn : '',
                    $record ? $fmtDate($record->date_of_appointment) : '',
                    $record ? $fmtDate($record->agm_held) : '',
                    $record ? $fmtDate($record->from_year) : '',
                    $record ? $fmtDate($record->to_year) : '',
                    $record ? $record->no_of_years : '',
                    $record ? $fmtDate($record->due_date) : '', // Assuming stored, or could calc: AGM + 15 days
                    $record ? $record->name_of_auditor_firm : '',
                    $record ? $record->name_of_signing_auditor : '',
                    $record ? $record->checked_by : '',
                ];

                fputcsv($file, $row);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
    public function exportForEdit(Request $request)
    {
        $this->authorize('viewAny', Adt1Compliance::class);

        // 1. Get Selected Session (or default to Active)
        $selectedSessionId = $request->get('session_id');
        if (!$selectedSessionId) {
            $activeSession = FinancialSession::where('is_active', true)->first();
            $selectedSessionId = $activeSession ? $activeSession->id : FinancialSession::latest()->first()->id;
        }

        // 2. Query Companies linked to this Session (SAME LOGIC AS INDEX)
        $query = Company::whereHas('financialSessions', function ($q) use ($selectedSessionId) {
            $q->where('financial_sessions.id', $selectedSessionId);
        })->with([
                    'adt1Compliances' => function ($q) use ($selectedSessionId) {
                        $q->where('financial_session_id', $selectedSessionId);
                    }
                ]);

        // Search Filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('cin', 'like', "%{$search}%");
            });
        }

        // Status Filter
        if ($request->filled('status')) {
            $status = $request->status;
            if ($status === 'Filed') {
                $query->whereHas('adt1Compliances', function ($q) use ($selectedSessionId) {
                    $q->where('financial_session_id', $selectedSessionId)
                        ->where('status', 'Filed');
                });
            } elseif ($status === 'Pending') {
                $query->where(function ($q) use ($selectedSessionId) {
                    $q->whereDoesntHave('adt1Compliances', function ($sub) use ($selectedSessionId) {
                        $sub->where('financial_session_id', $selectedSessionId);
                    })->orWhereHas('adt1Compliances', function ($sub) use ($selectedSessionId) {
                        $sub->where('financial_session_id', $selectedSessionId)
                            ->where('status', 'Pending');
                    });
                });
            } elseif ($status === 'Overdue') {
                $query->whereHas('adt1Compliances', function ($q) use ($selectedSessionId) {
                    $q->where('financial_session_id', $selectedSessionId)
                        ->where('status', 'Overdue');
                });
            }
        }

        $companies = $query->orderBy('name')->get();
        $session = FinancialSession::find($selectedSessionId);

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="adt1_edit_export_' . date('Y-m-d') . '.csv"',
            'Pragma' => 'no-cache',
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Expires' => '0',
        ];

        // STRICT COLUMNS FOR ROUND-TRIP IMPORT
        $columns = [
            'CIN',
            'status',
            'date_of_filing',
            'srn',
            'date_of_appointment',
            'agm_held',
            'from_year',
            'to_year',
            'no_of_years',
            'name_of_auditor_firm',
            'name_of_signing_auditor',
            'checked_by'
        ];

        $callback = function () use ($companies, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($companies as $company) {
                $record = $company->adt1Compliances->first();

                // Status defaults to Pending if no record
                $status = $record ? $record->status : 'Pending';

                // Helper to format date to d/m/Y or empty
                $fmt = fn($d) => $d ? $d->format('d/m/Y') : '';

                $row = [
                    $company->cin, // 0: CIN
                    $status,       // 1: status
                    $record ? $fmt($record->date_of_filing) : '', // 2: date_of_filing
                    $record ? $record->srn : '', // 3: srn
                    $record ? $fmt($record->date_of_appointment) : '', // 4: date_of_appointment
                    $record ? $fmt($record->agm_held) : '', // 5: agm_held
                    $record ? $fmt($record->from_year) : '', // 6: from_year
                    $record ? $fmt($record->to_year) : '', // 7: to_year
                    $record ? $record->no_of_years : '', // 8: no_of_years
                    $record ? $record->name_of_auditor_firm : '', // 9: name_of_auditor_firm
                    $record ? $record->name_of_signing_auditor : '', // 10: name_of_signing_auditor
                    $record ? $record->checked_by : '' // 11: checked_by
                ];

                fputcsv($file, $row);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
