<?php

namespace App\Services;

use App\Models\Compliances\Adt1Compliance;
use App\Models\Core\Company;
use App\Models\Core\FinancialSession;
use Illuminate\Support\Facades\DB;
use Exception;

class Adt1ComplianceService
{
    /**
     * Update or Create ADT-1 Compliance Record
     */
    public function updateCompliance(Company $company, FinancialSession $session, array $data): Adt1Compliance
    {
        $adt1Type = \App\Models\Core\ComplianceType::where('code', 'ADT1')->firstOrFail();

        // Convert Dates from d/m/Y to Y-m-d for Database Storage
        $dateFields = ['date_of_filing', 'date_of_appointment', 'agm_held', 'from_year', 'to_year'];
        foreach ($dateFields as $field) {
            if (!empty($data[$field])) {
                try {
                    $data[$field] = \Carbon\Carbon::createFromFormat('d/m/Y', $data[$field])->format('Y-m-d');
                } catch (\Exception $e) {
                    $data[$field] = null;
                }
            }
        }

        // Calculate Due Date Logic (Service expects Y-m-d format in $data now)
        // calculateDueDate will parse standard Y-m-d
        $data['due_date'] = $this->calculateDueDate($data);

        return Adt1Compliance::updateOrCreate(
            [
                'company_id' => $company->id,
                'financial_session_id' => $session->id,
                'compliance_type_id' => $adt1Type->id,
            ],
            [
                'status' => $data['status'],
                'date_of_filing' => $data['date_of_filing'] ?? null,
                'due_date' => $data['due_date'], // Derived
                'srn' => $data['srn'] ?? null,
                'date_of_appointment' => $data['date_of_appointment'] ?? null,
                'agm_held' => $data['agm_held'] ?? null,
                'from_year' => $data['from_year'] ?? null,
                'to_year' => $data['to_year'] ?? null,
                'no_of_years' => $data['no_of_years'] ?? null,
                'name_of_signing_auditor' => $data['name_of_signing_auditor'] ?? null,
                'name_of_auditor_firm' => $data['name_of_auditor_firm'] ?? null,
                'checked_by' => $data['checked_by'] ?? null,
            ]
        );
    }

    public function calculateDueDate(array $data): ?string
    {
        // Rule: Actual Due Date = To Date + 6 Months
        // This is the deadline for filing.
        // NOT the notification date.

        if (empty($data['to_year'])) {
            return null;
        }

        try {
            // data[to_year] is trusted to be Y-m-d here
            $toDate = \Carbon\Carbon::parse($data['to_year']);

            // Logic: Add 6 Months strictly
            return $toDate->addMonths(6)->format('Y-m-d');

        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Import ADT-1 records from CSV
     */
    public function importCompliances(string $filePath, int $sessionId): array
    {
        $activeSession = FinancialSession::findOrFail($sessionId);
        $adt1Type = \App\Models\Core\ComplianceType::where('code', 'ADT1')->firstOrFail();

        $imported = 0;
        $failed = 0;

        // Use DB Facade for nested transaction support
        DB::beginTransaction();

        $handle = null;

        try {
            $handle = fopen($filePath, 'r');
            if (!$handle) {
                throw new Exception("Could not open CSV file.");
            }

            // Remove header row
            fgetcsv($handle);

            while (($row = fgetcsv($handle)) !== false) {
                // Mapping (Index based on New Input-Name Header Layout)
                // 0: CIN
                // 1: status
                // 2: date_of_filing
                // 3: srn
                // 4: date_of_appointment
                // 5: agm_held
                // 6: from_year
                // 7: to_year
                // 8: no_of_years
                // 9: due_date
                // 9: name_of_auditor_firm (Was 10)
                // 10: name_of_signing_auditor (Was 11)

                $cin = trim($row[0] ?? '');
                if (empty($cin)) {
                    continue; // Skip empty rows
                }

                $company = Company::where('cin', $cin)->first();

                if ($company) {
                    // Ensure link to session
                    if (!$company->financialSessions()->where('financial_session_id', $sessionId)->exists()) {
                        $company->financialSessions()->attach($sessionId, ['status' => 'active']);
                    }

                    // Parse Status
                    $status = ucfirst(strtolower(trim($row[1] ?? 'Pending')));
                    if (!in_array($status, ['Filed', 'Pending', 'Overdue']))
                        $status = 'Pending';

                    // Parse Dates safely (Expecting d/m/Y in CSV)
                    $dateOfFiling = !empty($row[2]) ? $this->parseDateToDb($row[2]) : null;

                    $srn = $row[3] ?? null;

                    $dateOfAppointment = !empty($row[4]) ? $this->parseDateToDb($row[4]) : null;
                    $agmHeld = !empty($row[5]) ? $this->parseDateToDb($row[5]) : null;
                    $fromYear = !empty($row[6]) ? $this->parseDateToDb($row[6]) : null;
                    $toYear = !empty($row[7]) ? $this->parseDateToDb($row[7]) : null;

                    $noOfYears = (isset($row[8]) && trim($row[8]) !== '') ? trim($row[8]) : null;

                    // Column 9 and 10 are now Auditor Details
                    $nameFirm = !(empty($row[9])) ? trim($row[9]) : null;
                    $nameSigning = !(empty($row[10])) ? trim($row[10]) : null;
                    $checkedBy = !(empty($row[11])) ? trim($row[11]) : null;

                    // Prepare data for calculation
                    // IMPORTANT: Pass Y-m-d formatted dates for calculation consistency
                    $rowPayload = [
                        'status' => $status,
                        'date_of_filing' => $dateOfFiling, // Y-m-d
                        'srn' => $srn,
                        'date_of_appointment' => $dateOfAppointment, // Y-m-d
                        'agm_held' => $agmHeld, // Y-m-d
                        'from_year' => $fromYear, // Y-m-d
                        'to_year' => $toYear, // Y-m-d
                        'name_of_signing_auditor' => $nameSigning,
                        'name_of_auditor_firm' => $nameFirm,
                        'checked_by' => $checkedBy,
                        'no_of_years' => $noOfYears
                    ];

                    // Use common logic
                    $rowPayload['due_date'] = $this->calculateDueDate($rowPayload);

                    Adt1Compliance::updateOrCreate(
                        [
                            'company_id' => $company->id,
                            'financial_session_id' => $sessionId,
                            'compliance_type_id' => $adt1Type->id,
                        ],
                        $rowPayload // This contains Y-m-d values, which DB expects
                    );
                    $imported++;
                } else {
                    $failed++;
                }
            }

            fclose($handle);
            $handle = null;
            DB::commit();

            return ['success' => true, 'imported' => $imported, 'failed' => $failed];

        } catch (Exception $e) {
            DB::rollBack();
            if ($handle)
                fclose($handle);
            throw $e;
        }
    }

    private function parseDateToDb($dateStr)
    {
        if (empty($dateStr))
            return null;
        try {
            // Try d/m/Y first
            return \Carbon\Carbon::createFromFormat('d/m/Y', trim($dateStr))->format('Y-m-d');
        } catch (\Exception $e) {
            try {
                // Fallback to Y-m-d just in case, or return null
                return \Carbon\Carbon::parse(trim($dateStr))->format('Y-m-d');
            } catch (\Exception $e) {
                return null;
            }
        }
    }
}
