@extends('admin.layouts.app')

@section('title', 'Companies List')

@section('content')
    <div class="max-w-7xl mx-auto py-4 px-3">

        <!-- Header -->
        <div class="flex items-center justify-between mb-3">
            <h1 class="text-sm font-semibold text-gray-800 tracking-wide">
                Companies
            </h1>
            <div class="flex gap-2">
                <a href="{{ route('admin.companies.sample-csv') }}"
                    class="text-[11px] px-3 py-1 border border-green-600 bg-green-600 text-white rounded hover:bg-green-700">
                    ⬇ Download Sample CSV
                </a>
                <button onclick="document.getElementById('csvImportModal').classList.remove('hidden')"
                    class="text-[11px] px-3 py-1 border border-purple-600 bg-purple-600 text-white rounded hover:bg-purple-700">
                    📥 Import CSV
                </button>
                <a href="{{ route('admin.companies.create') }}"
                    class="text-[11px] px-3 py-1 border border-blue-600 bg-blue-600 text-white rounded hover:bg-blue-700">
                    + Add Company
                </a>
            </div>
        </div>

        <!-- Stats & Search Bar -->
        <div class="grid grid-cols-1 md:grid-cols-4 gap-3 mb-4">
            <!-- Stats -->
            <div class="bg-white border border-gray-300 rounded p-3 flex flex-col justify-center">
                <span class="text-[10px] uppercase font-semibold text-gray-500">Total Companies</span>
                <span class="text-xl font-bold text-gray-800">{{ $companies->total() }}</span>
            </div>
            <div class="bg-white border border-gray-300 rounded p-3 flex flex-col justify-center">
                <span class="text-[10px] uppercase font-semibold text-gray-500">Active Sessions</span>
                <span
                    class="text-xl font-bold text-green-600">{{ $companies->where('financialSessions_count', '>', 0)->count() }}</span>
            </div>

            <!-- Search -->
            <div class="md:col-span-2">
                <form method="GET" action="{{ route('admin.companies.index') }}" class="h-full">
                    <div class="flex h-full">
                        <input type="text" name="search" value="{{ request('search') }}"
                            placeholder="Search by Name, CIN, PAN..."
                            class="flex-1 h-full text-xs px-3 border border-gray-300 rounded-l focus:border-blue-500 focus:ring-0">
                        <button type="submit"
                            class="bg-gray-100 border border-l-0 border-gray-300 text-gray-600 px-4 rounded-r hover:bg-gray-200 text-xs font-semibold">
                            Search
                        </button>
                        @if(request('search'))
                            <a href="{{ route('admin.companies.index') }}"
                                class="ml-2 flex items-center justify-center px-3 border border-red-200 bg-red-50 text-red-600 rounded hover:bg-red-100 text-xs">
                                Clear
                            </a>
                        @endif
                    </div>
                </form>
            </div>
        </div>

        <!-- Main Card -->
        <div class="bg-white border border-gray-300 rounded-md shadow-sm">

            @if(session('success'))
                <div class="bg-green-50 text-green-800 text-xs px-4 py-2 border-b border-green-100">
                    {{ session('success') }}
                </div>
            @endif

            @if(session('error'))
                <div class="bg-red-50 text-red-800 text-xs px-4 py-2 border-b border-red-100">
                    {{ session('error') }}
                </div>
            @endif

            @if(session('import_errors') && count(session('import_errors')) > 0)
                <div class="bg-yellow-50 text-yellow-800 text-xs px-4 py-3 border-b border-yellow-100">
                    <p class="font-semibold mb-2">Import Errors:</p>
                    <ul class="list-disc pl-5 space-y-1">
                        @foreach(session('import_errors') as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            @endif

            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead class="bg-gray-50 border-b border-gray-200">
                        <tr>
                            <th class="px-4 py-2 text-left text-[10px] font-bold text-gray-600 uppercase tracking-wider">
                                Company</th>
                            <th class="px-4 py-2 text-left text-[10px] font-bold text-gray-600 uppercase tracking-wider">
                                Identifiers</th>
                            <th class="px-4 py-2 text-left text-[10px] font-bold text-gray-600 uppercase tracking-wider">
                                Sessions</th>
                            <th class="px-4 py-2 text-center text-[10px] font-bold text-gray-600 uppercase tracking-wider">
                                Actions</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100">
                        @forelse($companies as $company)
                            <tr class="hover:bg-slate-50">
                                <td class="px-4 py-2">
                                    <div class="text-xs font-semibold text-gray-900 uppercase">{{ $company->name }}</div>
                                    @if($company->email)
                                        <div class="text-[10px] text-gray-500">{{ $company->email }}</div>
                                    @endif
                                </td>
                                <td class="px-4 py-2">
                                    <div class="flex flex-col gap-1">
                                        @if($company->cin)
                                            <span class="text-[10px] text-gray-600"><span class="font-semibold">CIN:</span>
                                                {{ $company->cin }}</span>
                                        @endif
                                        @if($company->pan)
                                            <span class="text-[10px] text-gray-600"><span class="font-semibold">PAN:</span>
                                                {{ $company->pan }}</span>
                                        @endif
                                        @if($company->gstin)
                                            <span class="text-[10px] text-gray-600"><span class="font-semibold">GST:</span>
                                                {{ $company->gstin }}</span>
                                        @endif
                                    </div>
                                </td>
                                <td class="px-4 py-2">
                                    @if($company->financialSessions->count() > 0)
                                        <span
                                            class="inline-flex items-center px-1.5 py-0.5 rounded text-[10px] font-medium bg-green-100 text-green-800 border border-green-200">
                                            {{ $company->financialSessions->count() }} Active
                                        </span>
                                    @else
                                        <span
                                            class="inline-flex items-center px-1.5 py-0.5 rounded text-[10px] font-medium bg-gray-100 text-gray-600 border border-gray-200">
                                            None
                                        </span>
                                    @endif
                                </td>
                                <td class="px-4 py-2 text-center">
                                    <div class="flex items-center justify-center gap-2">
                                        <a href="{{ route('admin.companies.show', $company->id) }}"
                                            class="text-gray-500 hover:text-blue-600 transition" title="View">
                                            <svg class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                                            </svg>
                                        </a>
                                        <a href="{{ route('admin.companies.edit', $company->id) }}"
                                            class="text-gray-500 hover:text-blue-600 transition" title="Edit">
                                            <svg class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
                                            </svg>
                                        </a>
                                        <form action="{{ route('admin.companies.destroy', $company->id) }}" method="POST"
                                            class="inline" onsubmit="return confirm('Delete this company?');">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit" class="text-gray-500 hover:text-red-600 transition"
                                                title="Delete">
                                                <svg class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                                </svg>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="4" class="px-4 py-8 text-center text-xs text-gray-500">
                                    No companies found. <a href="{{ route('admin.companies.create') }}"
                                        class="text-blue-600 hover:underline">Add one now</a>.
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            @if($companies->hasPages())
                <div class="border-t border-gray-200 px-4 py-3 bg-gray-50">
                    {{ $companies->links() }}
                </div>
            @endif
        </div>

        <!-- CSV Import Modal -->
        <div id="csvImportModal"
            class="hidden fixed inset-0 bg-gray-900 bg-opacity-50 flex items-center justify-center z-50">
            <div class="bg-white rounded-lg shadow-xl max-w-md w-full mx-4">
                <div class="flex items-center justify-between border-b border-gray-200 px-6 py-4">
                    <h3 class="text-sm font-semibold text-gray-800">Import Companies from CSV</h3>
                    <button onclick="document.getElementById('csvImportModal').classList.add('hidden')"
                        class="text-gray-400 hover:text-gray-600">
                        <svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M6 18L18 6M6 6l12 12" />
                        </svg>
                    </button>
                </div>
                <form action="{{ route('admin.companies.import-csv') }}" method="POST" enctype="multipart/form-data">
                    @csrf
                    <div class="px-6 py-4">
                        <div class="mb-4">
                            <label class="block text-xs font-medium text-gray-700 mb-2">
                                Select CSV File <span class="text-red-500">*</span>
                            </label>
                            <input type="file" name="csv_file" accept=".csv" required
                                class="block w-full text-xs text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded file:border-0 file:text-xs file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100">
                        </div>
                        <div class="bg-blue-50 border border-blue-200 rounded p-3 text-xs text-blue-800">
                            <p class="font-semibold mb-1">📋 Instructions:</p>
                            <ul class="list-disc pl-4 space-y-1">
                                <li>Download the sample CSV to see the required format</li>
                                <li>Ensure all required fields are filled</li>
                                <li>Financial Year ID must be valid (1, 2, 3...)</li>
                                <li>Date format: YYYY-MM-DD</li>
                            </ul>
                        </div>
                    </div>
                    <div class="border-t border-gray-200 px-6 py-4 flex justify-end gap-2">
                        <button type="button" onclick="document.getElementById('csvImportModal').classList.add('hidden')"
                            class="px-4 py-2 text-xs font-medium text-gray-700 bg-white border border-gray-300 rounded hover:bg-gray-50">
                            Cancel
                        </button>
                        <button type="submit"
                            class="px-4 py-2 text-xs font-medium text-white bg-purple-600 border border-purple-600 rounded hover:bg-purple-700">
                            Import CSV
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
@endsection