<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Compliances\Adt1Compliance;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class SendAdt1DueNotifications extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'compliance:adt1-notify';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Send one-time notifications for ADT-1 Compliances due in the current FY';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting ADT-1 Notification Check...');

        // Logic:
        // Find Pending ADT-1 records where:
        // 1. notification_sent is FALSE
        // 2. to_year is NOT NULL (Trigger depends on this)
        // 3. Current Date >= 01/04 of To Date Year

        // This ensures the notification triggers exactly ONCE when the derived date is reached.

        $today = Carbon::today();

        $records = Adt1Compliance::where('status', 'Pending')
            ->where('notification_sent', false)
            ->whereNotNull('to_year') // We need to_year to calculate the Indicator Date (Trigger)
            ->with('company')
            ->get();

        $count = 0;

        foreach ($records as $record) {
            // New Logic: Trigger on 01/04 of the "To Date" Year
            // Example: To Date = 30/11/2026 -> Year is 2026
            // Trigger Date = 01/04/2026

            if (!$record->to_year) {
                continue;
            }

            // Create Trigger Date: 1st April of the To Year
            $triggerYear = $record->to_year->year;
            $triggerDate = Carbon::createFromDate($triggerYear, 4, 1)->startOfDay();

            // Check if Today is on or after the Trigger Date
            if ($today->greaterThanOrEqualTo($triggerDate)) {

                // SEND NOTIFICATION HERE
                // For now, we Log it and mark sent.
                // In a real app, this would trigger an Email/Notification class.

                $dueDateStr = optional($record->due_date)->format('d-m-Y') ?? 'N/A';
                Log::info("ADT-1 Notification Sent: Company {$record->company->name} (CIN: {$record->company->cin}). Due Date: {$dueDateStr}");

                $this->info("Notifying: {$record->company->name}");

                // Mark as sent to ensure ONE-TIME trigger
                $record->notification_sent = true;
                $record->save();

                $count++;
            }
        }

        $this->info("Processed {$count} notifications successfully.");
    }
}
